// Copyright (c) 2022 CTTC
// Copyright (c) 2023 Centre Tecnologic de Telecomunicacions de Catalunya (CTTC)
//
// SPDX-License-Identifier: GPL-2.0-only

#ifndef TRAFFIC_GENERATOR_NGMN_VIDEO_H
#define TRAFFIC_GENERATOR_NGMN_VIDEO_H

#include "traffic-generator.h"

#include "ns3/random-variable-stream.h"

namespace ns3
{

class TrafficGeneratorNgmnVideoTestCase;

/**
 * Video application that implements the video traffic model explained in
 * the Annex A of White Paper by the NGMN Alliance.
 * Basically, the video traffic is composed of frames.
 * Each frame of video data arrives at a regular interval T determined by
 * the number of frames per second. Each frame is decomposed into a fixed number
 * of slices, each transmitted as a single packet. The size of these
 * packets/slices is modeled to have a truncated Pareto distribution.
 * The video encoder introduces encoding delay intervals between the packets
 * of a frame. These intervals are modeled by a truncated Pareto
 * distribution.
 * The default configurations of distributions of packet size and packet time
 * assume a source video rate of 64 kbps.
 */

class TrafficGeneratorNgmnVideo : public TrafficGenerator
{
    friend TrafficGeneratorNgmnVideoTestCase;

  public:
    /**
     * @brief Get the type ID.
     * @return the object TypeId
     */
    static TypeId GetTypeId();

    TrafficGeneratorNgmnVideo();

    ~TrafficGeneratorNgmnVideo() override;

    /**
     * Assign a fixed random variable stream number to the random variables
     * used by this model. Return the number of streams (possibly zero) that
     * have been assigned.
     *
     * @param stream first stream index to use
     * @return the number of stream indices assigned by this model
     */
    int64_t AssignStreams(int64_t stream) override;

  protected:
    void DoDispose() override;
    void DoInitialize() override;

  private:
    // inherited from Application base class.
    // Called at time specified by Start by the DoInitialize method
    void StartApplication() override;

    // inherited from TrafficGenerator
    void PacketBurstSent() override;
    /**
     * @brief Get the amount of data to transfer
     * @return the amount of data to transfer
     */
    uint32_t GetNextPacketSize() const override;
    /**
     * @brief Get the relative time when the next packet should be sent
     * @return the relative time when the next packet will be sent
     */
    Time GetNextPacketTime() const override;
    /**
     * @brief Generates the frame length in number of packets
     */
    void GenerateNextPacketBurstSize() override;

    uint32_t m_numberOfPacketsInFrame{0}; //!< Number of packets in a frame
    Time m_interframeIntervalTime{0};     //!< Interframe interval time
    Ptr<ParetoRandomVariable>
        m_packetSizeGenerator; //!< Pareto random variable for packet size generation
    Ptr<ParetoRandomVariable>
        m_packetTimeGenerator; //!< Pareto random variable for packet time generation
    double m_packetSizeScale{
        0.0}; //!< The scale parameter for the Pareto distribution for the packet size generation
    double m_packetSizeShape{
        0.0}; //!< The shape parameter for the Pareto distribution for the packet size generation
    double m_packetSizeBound{
        0.0}; //!< The bound parameter for the Pareto distribution for the packet size generation
    double m_packetTimeScale{
        0.0}; //!< The scale parameter for the Pareto distribution for the packet time generation
    double m_packetTimeShape{
        0.0}; //!< The shape parameter for the Pareto distribution for the packet time generation
    double m_packetTimeBound{
        0.0}; //!< The bound parameter for the Pareto distribution for the packet time generation
    uint32_t m_packetFrameCounter{0}; //!< The packet frame counter generated by this application
    static uint32_t m_flowIdCounter;  //!< The global flow ID counter of video traffic generators
    uint32_t m_flowId{0};             //!< The traffic generator flow Id of this instance
};

} // namespace ns3

#endif /* TRAFFIC_GENERATOR_NGMN_VIDEO_H */
