/*****************************************************************************
 * pmnotebook.c
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * Copyright (C) 2006, Greg Hosler
 * (ghosler ['at'] users.sourceforge.net)
 * 
 * Released under the terms of the GPL.
 * *NO WARRANTY*
 *
 *****************************************************************************/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <gtk/gtk.h>
#include <glib-object.h>

#include "config.h"

#include "users.h"
#include "util.h"
#include "gyachi_notebook.h"
#include "pmnotebook.h"
#include "callbacks.h"
#include "packet_handler.h"
#include "gyach.h"
#include "main.h"

#include "theme_support.h"
#include "gy_config.h"
#include "gyachi_lib.h"


/* forward declarations */
static gboolean pm_notebook_close_win_cb(GtkWidget *w, GdkEventFocus *e, gpointer d);
static gboolean pm_notebook_focus_win_cb(GtkWidget *w, GdkEventFocus *e, gpointer d);
static gboolean pm_notebook_unfocus_win_cb(GtkWidget *w, GdkEventFocus *e, gpointer d);
static gboolean pm_notebook_page_removed_cb(GtkWidget *notebook, GtkWidget *page, int page_num, gpointer d);
static void     pm_notebook_before_switch_page_cb(GtkWidget *notebook, GtkWidget *page, int page_num, gpointer d);
static void     pm_notebook_switch_page_cb(GtkWidget *notebook, GtkWidget *page, int page_num, gpointer d);


/**************************************************************************
 * GTK+ notebook and window ops
 **************************************************************************/
GYACHI_NOTEBOOK_INTERFACE *conversation=NULL;

void pm_notebook_save()
{
	if (remember_notebook_position) {
	        if (conversation) {
			gtk_window_get_position(GTK_WINDOW(conversation->window),
					&notebook_window_x, &notebook_window_y );

			/* get current sizes */
			notebook_window_width  = conversation->window->allocation.width;
			notebook_window_height = conversation->window->allocation.height;
		}
	}
	else {
	        notebook_window_x = 0;
		notebook_window_y = 0;
		notebook_window_width  = 512;
		notebook_window_height = 365;
	}
}

GYACHI_NOTEBOOK_INTERFACE *pm_notebook_create()
{
        GYACHI_NOTEBOOK_INTERFACE *gy_notebook=NULL;
	GtkWidget *vbox;

	gy_notebook = gyachi_notebook_new(gtk_window_new(GTK_WINDOW_TOPLEVEL));

	/*
	 * gtk_window_set_role(GTK_WINDOW(gy_notebook->window), "conversation");
	 */
	if (remember_notebook_position) {
		if (notebook_window_width  == 0) notebook_window_width  = 512;
		if (notebook_window_height == 0) notebook_window_height = 365;
		gtk_window_set_default_size (GTK_WINDOW(gy_notebook->window), notebook_window_width, notebook_window_height);
		gtk_window_move(GTK_WINDOW(gy_notebook->window),notebook_window_x, notebook_window_y);
	}
	else {
		gtk_window_set_default_size (GTK_WINDOW(gy_notebook->window), 512, 365);
		gtk_window_move(GTK_WINDOW(gy_notebook->window), 0, 0);
	}

	gtk_window_set_type_hint(GTK_WINDOW(gy_notebook->window), GDK_WINDOW_TYPE_HINT_NORMAL);
	gtk_window_set_resizable(GTK_WINDOW(gy_notebook->window), TRUE);
	gtk_container_set_border_width(GTK_CONTAINER(gy_notebook->window), 0);
	GTK_WINDOW(gy_notebook->window)->allow_shrink = TRUE;

	g_signal_connect(G_OBJECT(gy_notebook->window), "delete_event",
			 G_CALLBACK(pm_notebook_close_win_cb), gy_notebook);
	g_signal_connect(G_OBJECT(gy_notebook->window), "focus_in_event",
			 G_CALLBACK(pm_notebook_focus_win_cb), gy_notebook);
	g_signal_connect(G_OBJECT(gy_notebook->window), "focus_out_event",
			 G_CALLBACK(pm_notebook_unfocus_win_cb), gy_notebook);

	gtk_widget_show(gy_notebook->notebook);
	g_signal_connect(G_OBJECT(gy_notebook->notebook), "switch_page",
			 G_CALLBACK(pm_notebook_before_switch_page_cb), gy_notebook);
	g_signal_connect_after(G_OBJECT(gy_notebook->notebook), "switch_page",
			       G_CALLBACK(pm_notebook_switch_page_cb), gy_notebook);
	g_signal_connect(G_OBJECT(gy_notebook->notebook), "page_removed",
			 G_CALLBACK(pm_notebook_page_removed_cb), gy_notebook);

	/* Setup the container. */
	vbox = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), gy_notebook->notebook, TRUE, TRUE, 0);
	gtk_container_add(GTK_CONTAINER(gy_notebook->window), vbox);
	gtk_widget_show(vbox);

	return(gy_notebook);
}

GYACHI_NOTEBOOK_INTERFACE *pm_notebook_add_window(GtkWidget *pm_window, char *label_text)
{
        GYACHI_NOTEBOOK_INTERFACE *gy_notebook=NULL;
	GtkWidget *hbox;
        GtkWidget *label;
        GtkWidget *menu_label;
	int       new_page;
	GtkWidget *image = NULL;

	if (pm_in_notebook) {
	        if (conversation == 0) {
		        conversation = pm_notebook_create();
		}
		gy_notebook = conversation;
	}
	else {
	        gy_notebook = pm_notebook_create();
	}

	menu_label = gtk_label_new(label_text);

	hbox = gtk_hbox_new(FALSE, 0);
	label = gtk_label_new(label_text);
	if (enable_tab_icons) {
		image = gtk_image_new_from_stock(GYACHI_ICON, GTK_ICON_SIZE_MENU);
		if (image) {
			gtk_box_pack_start(GTK_BOX(hbox), image, FALSE, FALSE, 0);
		}
	}
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_box_set_spacing(GTK_BOX (hbox), 4);

	/*
	 * store both a pointer to the conversation box, as well as a
	 * pointer to tab, in the pm_session UI
	 *
	 * NOTE: these set_data's NEED to be before the append page.
	 *       append page will invoke the set_title method (below)
	 *       which will need these values set.
	 */
	g_object_set_data(G_OBJECT(pm_window), "tab_label",  label);
	g_object_set_data(G_OBJECT(pm_window), "notebook",   gy_notebook);
	g_object_set_data(G_OBJECT(pm_window), "label_hbox", hbox);
	g_object_set_data(G_OBJECT(pm_window), "hbox-image", image);

	gtk_widget_show(pm_window);
	gtk_widget_show(label);
	gtk_widget_show_all(hbox);

	new_page = gtk_notebook_append_page_menu(GTK_NOTEBOOK(gy_notebook->notebook), pm_window, hbox, menu_label);
	gtk_widget_show(gy_notebook->window);
	return(gy_notebook);
}

void pm_notebook_set_title(GYACHI_NOTEBOOK_INTERFACE *gy_notebook)
{
        GtkWidget *pm_window;
	GdkPixbuf *buf  = NULL;
	GtkWidget *image;
	PM_SESSION *pm_sess;
	int page;
	int need_to_scale = 0;
	char titlebf[128];

	page = gtk_notebook_get_current_page(GTK_NOTEBOOK(gy_notebook->notebook));
	pm_window = gtk_notebook_get_nth_page(GTK_NOTEBOOK(gy_notebook->notebook), page);
	pm_sess = g_object_get_data(G_OBJECT(pm_window), "session");
	image   = g_object_get_data(G_OBJECT(pm_window), "hbox-image");

	if (image) {
		gtk_widget_destroy(image);
		image = NULL;
	}


	/* buffer overflow limit set */
	snprintf( titlebf, 124, "%s%s - GyachE%s", (gy_notebook == conversation)?"* ":"",
		  pm_sess->pm_user,
		  (gy_notebook == conversation)?" *":"");

	gtk_window_set_title(GTK_WINDOW(gy_notebook->window), titlebf);

	if (pm_sess == NULL) {
	        /*
		 * this will happen the very first time we add the first page to the
		 * conversation notebook. The page is created, and added, before the
		 * session pointer is in the widget. Just ignore this case!
		 */
	        return;
	}

	if (show_bicon) {
		GtkWidget *pm_panel;
		int        show_icon = 1;

		pm_panel = g_object_get_data(G_OBJECT(pm_window), "panel");
		if (pm_panel && g_object_get_data(G_OBJECT(pm_panel), "hide")) {
			show_icon=0;
		}

		if ((pm_sess->bimage_them) && show_icon) {
			GtkImageType type;
			
			type = gtk_image_get_storage_type(GTK_IMAGE(pm_sess->bimage_them));
			if (type == GTK_IMAGE_PIXBUF) {
				buf = gtk_image_get_pixbuf(GTK_IMAGE(pm_sess->bimage_them));

				if (buf) {
					need_to_scale = 1;
					g_object_ref(buf);
				}
			}
		}
	}
	if (!buf) {
		buf = get_pixbuf_from_stock_id(GTK_WIDGET(gy_notebook->window),
					       GYACHI_ICON,
					       GTK_ICON_SIZE_MENU);
	}

	gtk_window_set_icon(GTK_WINDOW(gy_notebook->window), buf);

	if (enable_tab_icons) {
		GdkPixbuf *buf2 = NULL;
		GtkWidget *hbox;

		hbox    = g_object_get_data(G_OBJECT(pm_window), "label_hbox");
		if (need_to_scale) {
			buf2 = gdk_pixbuf_scale_simple(buf, 16,16, GDK_INTERP_BILINEAR );
			g_object_unref(buf);
			buf = buf2;
			buf2 = NULL;
		}

		image = gtk_image_new_from_pixbuf (buf);
		gtk_box_pack_start(GTK_BOX(hbox), image, FALSE, FALSE, 0);
		gtk_box_reorder_child(GTK_BOX(hbox), image, 0);
		gtk_widget_show_all(hbox);
	}

	g_object_set_data(G_OBJECT(pm_window), "hbox-image", image);

	g_object_unref(buf);
}

void pm_notebook_focus_page(PM_SESSION *pm_sess)
{
	if (pm_sess) {
	        pm_notebook_select_tab(pm_sess->pm_notebook, pm_sess->pm_user);
	}
}

char *escape_html(const char *html) {
	const char *c = html;
	GString *ret;

	if (html == NULL)
		return NULL;

	ret = g_string_new("");

	while (*c) {
		switch (*c) {
			case '&':
				ret = g_string_append(ret, "&amp;");
				break;
			case '<':
				ret = g_string_append(ret, "&lt;");
				break;
			case '>':
				ret = g_string_append(ret, "&gt;");
				break;
			case '"':
				ret = g_string_append(ret, "&quot;");
				break;
			default:
				ret = g_string_append_c(ret, *c);
		}
		c++;
	}

	return g_string_free(ret, FALSE);
}

static char *make_markup_string(char *username, char *color)
{
        char buff[512];
	char *html_username = escape_html(username);
	snprintf(buff, 512, "<span color=\"%s\">%s</span>", color, html_username);
	g_free(html_username);
	return(strdup(buff));
}

/*
 */
void pm_notebook_set_flashing_status(PM_SESSION *pm_sess, int status)
{
	GYACHI_NOTEBOOK_INTERFACE *gy_notebook;

	gy_notebook=g_object_get_data(G_OBJECT(pm_sess->pm_window), "notebook");
	if (status) {
		if (!gtk_window_is_active(GTK_WINDOW(gy_notebook->window))) {
			gtk_window_set_urgency_hint(GTK_WINDOW(gy_notebook->window), TRUE);
		}
	}
	else {
		gtk_window_set_urgency_hint(GTK_WINDOW(gy_notebook->window), FALSE);
	}
}


/*
 * set and/or check, and as necessary set, typing status in
 * the notebook label, as well as the pm box itself.
 * TYPING_STATUS:
 *         TYPING:
 *                  set [TYPING] in pms_text box
 *                  set tab color
 *
 *         TYPED:
 *                  clear pms_text box
 *
 *         UNSEEN_TEXT:
 *                  clear pms_text box
 *                  if tab is curent tab, and in focus, then reset color to black.
 *                  otherwise set color to UNSEEN_TEXT color.
 *
 *         SENT_TEXT:
 *                  like UNSEEN_TEXT, but do not initialize typing status.
 *
 * typing_status:
 *         <no value> (or 0)      : uninitialized
 *         1                        incoming text (typing)
 *         2                        incoming text completed.
 * event_status:
 *         <no value> (or 0)      : uninitialized
 *         1                        unseen event
 */
void pm_notebook_set_typing_status(PM_SESSION *pm_sess, TYPING_STATUS status)
{
	GYACHI_NOTEBOOK_INTERFACE *gy_notebook;
	GtkWidget *pms_status;
	GtkWidget *tab_label;
	GtkWidget *pm_window;
	char typing_text[300];
	char *label;
	int typing_status;
	int event_status;
	int page;

	if (pm_sess == NULL) return;

	gy_notebook = pm_sess->pm_notebook;
	page = gtk_notebook_get_current_page(GTK_NOTEBOOK(gy_notebook->notebook));
	pm_window = gtk_notebook_get_nth_page(GTK_NOTEBOOK(gy_notebook->notebook), page);

	if (pm_sess) {
	        pms_status=g_object_get_data(G_OBJECT(pm_sess->pm_window), "pms_status");
		tab_label =g_object_get_data(G_OBJECT(pm_sess->pm_window), "tab_label");

		typing_status = (int)g_object_get_data(G_OBJECT(tab_label), "typing_status");
		event_status  = (int)g_object_get_data(G_OBJECT(tab_label), "event_status");
	}
	else {
	        pms_status=NULL;
		tab_label=NULL;
		typing_status=0;
		event_status =0;
	}

	switch (status) {
	case TYPING:
	        if (pms_status) {
		        snprintf(typing_text, 300, "<span foreground=\"#a04dd1\"  font_desc=\"sans bold 12\" background=\"#e8eef7\">[</span><span foreground=\"#4251b6\"  font_desc=\"sans bold 12\" background=\"#e8eef7\">%s</span><span foreground=\"#a04dd1\"  font_desc=\"sans bold 12\" background=\"#e8eef7\">]</span>", _("[TYPING]"));
			gtk_label_set_markup(GTK_LABEL(pms_status), typing_text);
			gtk_widget_show_all(pms_status);
		}
		if (tab_label) {
		        label=make_markup_string(pm_sess->pm_user, TEXT_GREEN);
			gtk_label_set_markup(GTK_LABEL(tab_label), label);
			free(label);
			g_object_set_data(G_OBJECT(tab_label), "typing_status", (gpointer)1);
		}
		break;
	  
	case TYPED:
	        if (pms_status) {
			gtk_label_set_text(GTK_LABEL(pms_status), "      ");
			gtk_widget_show_all(pms_status);
		}
		break;

	case UNSEEN_TEXT:
	        if (pms_status) {
			gtk_label_set_text(GTK_LABEL(pms_status), "      ");
			gtk_widget_show_all(pms_status);
		}
		if (tab_label) {
		        typing_status = (int)g_object_get_data(G_OBJECT(tab_label), "typing_status");
			if (gy_notebook->has_focus && (pm_sess->pm_window == pm_window)) {
				label=make_markup_string(pm_sess->pm_user, TEXT_BLACK);
				gtk_label_set_markup(GTK_LABEL(tab_label), label);
				free(label);
			}
			else {
			        if (event_status) {
				        label=make_markup_string(pm_sess->pm_user, TEXT_BLUE);
				}
				else {
				        label=make_markup_string(pm_sess->pm_user, TEXT_ORANGE);
				}
				gtk_label_set_markup(GTK_LABEL(tab_label), label);
				free(label);
			}
			g_object_set_data(G_OBJECT(tab_label), "typing_status", (gpointer)2);
		}
		break;

	case UNSEEN_EVENT:
		if (tab_label) {
		        typing_status = (int)g_object_get_data(G_OBJECT(tab_label), "typing_status");
			if (gy_notebook->has_focus && (pm_sess->pm_window == pm_window)) {
				label=make_markup_string(pm_sess->pm_user, TEXT_BLACK);
				gtk_label_set_markup(GTK_LABEL(tab_label), label);
				free(label);
			}
			else {
				label=make_markup_string(pm_sess->pm_user, TEXT_BLUE);
				gtk_label_set_markup(GTK_LABEL(tab_label), label);
				free(label);
			}
			g_object_set_data(G_OBJECT(tab_label), "event_status", (gpointer)1);
			if (typing_status == 1) {
			        g_object_set_data(G_OBJECT(tab_label), "typing_status", (gpointer)2);
			}
		}
		break;

	case SENT_TEXT:
		if (tab_label) {
		        typing_status = (int)g_object_get_data(G_OBJECT(tab_label), "typing_status");
			if (typing_status == 1) {
				label=make_markup_string(pm_sess->pm_user, TEXT_GREEN);
				gtk_label_set_markup(GTK_LABEL(tab_label), label);
				free(label);
			}
			else {
			        if (gy_notebook->has_focus && (pm_sess->pm_window == pm_window)) {
				        label=make_markup_string(pm_sess->pm_user, TEXT_BLACK);
					gtk_label_set_markup(GTK_LABEL(tab_label), label);
					free(label);
					g_object_set_data(G_OBJECT(tab_label), "event_status", (gpointer)0);
				}
				else {
				        if (event_status) {
					        label=make_markup_string(pm_sess->pm_user, TEXT_BLUE);
					}
					else {
					        label=make_markup_string(pm_sess->pm_user, TEXT_GREEN);
					}
					gtk_label_set_markup(GTK_LABEL(tab_label), label);
					free(label);
				}
			}
		}
		break;


	case SELECT_PAGE:
		if (tab_label) {
		        typing_status = (int)g_object_get_data(G_OBJECT(tab_label), "typing_status");
			if (((typing_status == 0) || (typing_status == 2)) &&
			    (gy_notebook->has_focus) && (pm_sess->pm_window == pm_window)) {
			        if (pms_status) {
				        gtk_label_set_text(GTK_LABEL(pms_status), "      ");
					gtk_widget_show_all(pms_status);
				}
				label=make_markup_string(pm_sess->pm_user, TEXT_BLACK);
				gtk_label_set_markup(GTK_LABEL(tab_label), label);
				free(label);
			}
			g_object_set_data(G_OBJECT(tab_label), "event_status", (gpointer)0);
		}
		break;

	case FOCUS:
		if (tab_label) {
		        typing_status = (int)g_object_get_data(G_OBJECT(tab_label), "typing_status");
			if ((typing_status == 0) ||
			    ((gy_notebook->has_focus) && (pm_sess->pm_window == pm_window) &&
			     (typing_status == 2))) {
			        if (pms_status) {
				        gtk_label_set_text(GTK_LABEL(pms_status), "      ");
					gtk_widget_show_all(pms_status);
				}
				label=make_markup_string(pm_sess->pm_user, TEXT_BLACK);
				gtk_label_set_markup(GTK_LABEL(tab_label), label);
				free(label);
			}
			else {
				label=make_markup_string(pm_sess->pm_user, TEXT_GREEN);
				gtk_label_set_markup(GTK_LABEL(tab_label), label);
				free(label);
			}
			g_object_set_data(G_OBJECT(tab_label), "event_status", (gpointer)0);
		}
		break;

	default:
		break;	   
	}
}

void pm_notebook_destroy(GYACHI_NOTEBOOK_INTERFACE *gy_notebook)
{
	pm_notebook_save();
	gyachi_notebooks = g_slist_remove(gyachi_notebooks, gy_notebook);
	gtk_widget_destroy(gy_notebook->notebook);
	gtk_widget_destroy(gy_notebook->window);
	free(gy_notebook);
	if (gy_notebook == conversation) {
	        conversation = 0;
	}
}

static void pm_notebook_delete_if_empty(GYACHI_NOTEBOOK_INTERFACE *gy_notebook)
{
        if (gtk_notebook_get_n_pages(GTK_NOTEBOOK(gy_notebook->notebook)) == 0) {
	        pm_notebook_destroy(gy_notebook);
	}
}

void pm_notebook_select_tab(GYACHI_NOTEBOOK_INTERFACE *gy_notebook, char *name)
{
        int npages    = gtk_notebook_get_n_pages(GTK_NOTEBOOK(gy_notebook->notebook));
	int curr_page = gtk_notebook_get_current_page(GTK_NOTEBOOK(gy_notebook->notebook));
	int counter;
	GtkWidget *page;
	GtkWidget *hbox;
	GList     *hbox_children;
	GList     *child;
	const char *label_text;

	for (counter=0; counter < npages; counter++) {
	        page = gtk_notebook_get_nth_page(GTK_NOTEBOOK(gy_notebook->notebook), counter);
		hbox = gtk_notebook_get_tab_label(GTK_NOTEBOOK(gy_notebook->notebook), page);
		hbox_children = gtk_container_get_children(GTK_CONTAINER(hbox));
		for (child = hbox_children; child; child=g_list_next(child)) {
			if (GTK_IS_LABEL(child->data)) {
				label_text = gtk_label_get_text(child->data);
				if (label_text && (strcmp(label_text, name) == 0)) {
					if (counter != curr_page) {
						gtk_notebook_set_current_page(GTK_NOTEBOOK(gy_notebook->notebook), counter);
					}
					pm_notebook_set_title(gy_notebook);
					g_list_free(hbox_children);
					return;
				}
			}
		}
		g_list_free(hbox_children);
	}
}

static gboolean pm_notebook_close_win_cb(GtkWidget *w, GdkEventFocus *e, gpointer d)
{
	GYACHI_NOTEBOOK_INTERFACE *gy_notebook = (GYACHI_NOTEBOOK_INTERFACE *)d;
	int npages = gtk_notebook_get_n_pages(GTK_NOTEBOOK(gy_notebook->notebook));
	int counter;
	GtkWidget *pm_window;
	GtkWidget *pms_close;

	pm_notebook_save();
	for (counter=npages-1; counter >= 0 && gy_notebook; counter--) {
	        pm_window = gtk_notebook_get_nth_page(GTK_NOTEBOOK(gy_notebook->notebook), counter);
		pms_close = g_object_get_data(G_OBJECT(pm_window), "pms_close");

		if (pms_close) {
		        on_pms_close_clicked(pms_close, NULL);
		}
	}

	/*
	 * no need to call pm_notebook_destroy() as 
	 * on_pms_close_clicked() will call shutdown_pms_session()
	 * which will eventually call pm_notebook_delete_if_empty().
	 * Removing the last page, will ultimately destroy the window.
	 */

	return(0);
}

static gboolean pm_notebook_page_removed_cb(GtkWidget *notebook, GtkWidget *page, int page_num, gpointer d)
{
	GYACHI_NOTEBOOK_INTERFACE *gy_notebook = (GYACHI_NOTEBOOK_INTERFACE *)d;

	pm_notebook_delete_if_empty(gy_notebook);
	return(0);
}

static gboolean pm_notebook_unfocus_win_cb(GtkWidget *w, GdkEventFocus *e, gpointer d)
{
	GYACHI_NOTEBOOK_INTERFACE *gy_notebook = (GYACHI_NOTEBOOK_INTERFACE *)d;

	gy_notebook->has_focus = FALSE;

	return(0);
}

static gboolean pm_notebook_focus_win_cb(GtkWidget *w, GdkEventFocus *e, gpointer d)
{
	GYACHI_NOTEBOOK_INTERFACE *gy_notebook = (GYACHI_NOTEBOOK_INTERFACE *)d;
	PM_SESSION *pm_sess;
	GtkWidget *pm_window;
	int page;

	gtk_window_set_urgency_hint(GTK_WINDOW(gy_notebook->window), FALSE);
	gy_notebook->has_focus = TRUE;

	page = gtk_notebook_get_current_page(GTK_NOTEBOOK(gy_notebook->notebook));
	pm_window = gtk_notebook_get_nth_page(GTK_NOTEBOOK(gy_notebook->notebook), page);

	pm_sess = g_object_get_data(G_OBJECT(pm_window),"session");
	pm_notebook_set_typing_status(pm_sess, FOCUS);

	return(FALSE);
}

/* When a notebook tab is clicked, 1st pm_notebook_before_switch_page_cb() will be
 * called on the old tab (the one we're leaving), and then
 * pm_notebook_switch_page_cb() will be called on the clicked tab.
 */
static void pm_notebook_before_switch_page_cb(GtkWidget *notebook, GtkWidget *page, int page_num, gpointer d)
{
	return;
}

static void pm_notebook_switch_page_cb(GtkWidget *n, GtkWidget *p, int page_num, gpointer d)
{
	GYACHI_NOTEBOOK_INTERFACE *gy_notebook = (GYACHI_NOTEBOOK_INTERFACE *)d;
	PM_SESSION *pm_sess;
	GtkWidget *pm_window;
	GtkWidget *pm_panel;
	GtkWidget *pm_dlmanager;

	pm_window    = gtk_notebook_get_nth_page(GTK_NOTEBOOK(gy_notebook->notebook), page_num);
	pm_sess      = g_object_get_data(G_OBJECT(pm_window), "session");
	pm_panel     = g_object_get_data(G_OBJECT(pm_window), "panel");
	pm_dlmanager = g_object_get_data(G_OBJECT(pm_window), "dlmanager");

	if (pm_sess == NULL) {
	        /*
		 * this will happen the very first time we add the first page to the
		 * conversation notebook. The page is created, and added, before the
		 * session pointer is in the widget. Just ignore this case!
		 */
		return;
	}

	pm_notebook_set_typing_status(pm_sess, SELECT_PAGE);
	gtk_widget_show_all(pm_window );
	if (pm_panel && g_object_get_data(G_OBJECT(pm_panel), "hide")) {
		gtk_widget_hide((GtkWidget *)pm_panel);
	}
	if (pm_dlmanager && g_object_get_data(G_OBJECT(pm_dlmanager), "hide")) {
		gtk_widget_hide((GtkWidget *)pm_dlmanager);
	}

	pm_notebook_set_title(gy_notebook);
	focus_pm_entry(pm_sess);

	return;
}

