/*****************************************************************************
 * gyachi-alsa.c, Plugin to use ALSA as the play/record device.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * Released under the terms of the GPL.
 * *NO WARRANTY*
 *
 * Copyright (C) 2008, Gregory D Hosler (ghosler ['at'] users.sourceforge.net)
 * Released under the terms of the GPL.
 *
 *****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <alsa/asoundlib.h>

#include <gtk/gtk.h>

#include "plugin_api.h"
#include "config.h"
#include "sound_plugin.h"


int alsa_frame_width(int format) {
	switch (format) {
	case SND_PCM_FORMAT_S8:
	case SND_PCM_FORMAT_U8:
	case SND_PCM_FORMAT_MU_LAW:
	case SND_PCM_FORMAT_A_LAW:
		return 1;

	case SND_PCM_FORMAT_S16_LE:
	case SND_PCM_FORMAT_S16_BE:
	case SND_PCM_FORMAT_U16_LE:
	case SND_PCM_FORMAT_U16_BE:
		return 2;

	case SND_PCM_FORMAT_S24_3LE:
	case SND_PCM_FORMAT_S24_3BE:
	case SND_PCM_FORMAT_U24_3LE:
	case SND_PCM_FORMAT_U24_3BE:
	case SND_PCM_FORMAT_S20_3LE:
	case SND_PCM_FORMAT_S20_3BE:
	case SND_PCM_FORMAT_U20_3LE:
	case SND_PCM_FORMAT_U20_3BE:
	case SND_PCM_FORMAT_S18_3LE:
	case SND_PCM_FORMAT_S18_3BE:
	case SND_PCM_FORMAT_U18_3LE:
	case SND_PCM_FORMAT_U18_3BE:
		return 3;

	case SND_PCM_FORMAT_S24_LE:
	case SND_PCM_FORMAT_S24_BE:
	case SND_PCM_FORMAT_U24_LE:
	case SND_PCM_FORMAT_U24_BE:
	case SND_PCM_FORMAT_S32_LE:
	case SND_PCM_FORMAT_S32_BE:
	case SND_PCM_FORMAT_U32_LE:
	case SND_PCM_FORMAT_U32_BE:
	case SND_PCM_FORMAT_FLOAT_LE:
	case SND_PCM_FORMAT_FLOAT_BE:
		return 4;

	case SND_PCM_FORMAT_FLOAT64_LE:
	case SND_PCM_FORMAT_FLOAT64_BE:
		return 8;

	/* the following are unknown sizes... */
	case SND_PCM_FORMAT_UNKNOWN:
	case SND_PCM_FORMAT_IEC958_SUBFRAME_LE:
	case SND_PCM_FORMAT_IEC958_SUBFRAME_BE:
	case SND_PCM_FORMAT_IMA_ADPCM:
	case SND_PCM_FORMAT_MPEG:
	case SND_PCM_FORMAT_GSM:
	case SND_PCM_FORMAT_SPECIAL:
	default:
		return 1; /* almost certainly wrong */
	}
}

/* map a GyachI format type to a ALSA format type */
int alsa_format(GYACHI_FORMAT_TYPE format) {
	switch (format) {
	case GY_SAMPLE_U8:		/* Unsigned 8 Bit PCM.	*/
		return(SND_PCM_FORMAT_U8);

	case GY_SAMPLE_ALAW:		/* 8 Bit a-Law		*/
		return (SND_PCM_FORMAT_A_LAW);

	case GY_SAMPLE_ULAW:		/* 8 Bit mu-Law		*/
		return(SND_PCM_FORMAT_MU_LAW);

	case GY_SAMPLE_S16LE:		/* Signed 16 Bit PCM, little endian (PC).	*/
		return(SND_PCM_FORMAT_S16_LE);

	case GY_SAMPLE_S16BE:		/* Signed 16 Bit PCM, big endian.		*/
		return(SND_PCM_FORMAT_S16_BE);

	case GY_SAMPLE_FLOAT32LE:	/* 32 Bit IEEE floating point, little endian, range -1 to 1	*/
		return(SND_PCM_FORMAT_FLOAT_LE);

	case GY_SAMPLE_FLOAT32BE:	/* 32 Bit IEEE floating point, big endian, range -1 to 1	*/
		return(SND_PCM_FORMAT_FLOAT_BE);

	case GY_SAMPLE_S32LE:		/* Signed 32 Bit PCM, little endian (PC).	*/
		return(SND_PCM_FORMAT_S32_LE);

	case GY_SAMPLE_S32BE:		/* Signed 32 Bit PCM, big endian (PC).		*/
		return(SND_PCM_FORMAT_S32_BE);

	default:
		return(SND_PCM_FORMAT_UNKNOWN);
	}
}

snd_pcm_stream_t alsa_stream(GYACHI_STREAM_TYPE stream_type) {
	switch (stream_type) {
	case GY_STREAM_PLAYBACK:
		return(SND_PCM_STREAM_PLAYBACK);

	case GY_STREAM_RECORD:
		return(SND_PCM_STREAM_CAPTURE);

	default:
		return(-1);
	}
}


void *alsa_open_device(GYACHI_STREAM_TYPE stream_type,
		       GYACHI_FORMAT_TYPE format_type,
		       int channels,
		       int rate) {
	static unsigned period_time = 0;
	static unsigned buffer_time = 0;
	snd_pcm_uframes_t period_frames = 0;
	snd_pcm_uframes_t buffer_frames = 0;
	snd_pcm_format_t format = alsa_format(format_type);
	snd_pcm_hw_params_t *hwparams;
	snd_pcm_t *pcm_handle;
	snd_pcm_stream_t stream = alsa_stream(stream_type);
	char *pcm_name = "default";
	int exact_rate;   // Sample rate returned by snd_pcm_hw_params_set_rate_near
	int err;

	snd_pcm_hw_params_alloca(&hwparams);

	if (snd_pcm_open(&pcm_handle, pcm_name, stream, 0) < 0) {
        	fprintf(stderr, "Error opening PCM device %s\n", pcm_name);
		return(NULL);
	}
  
	/* Init hwparams with full configuration space */
	if (snd_pcm_hw_params_any(pcm_handle, hwparams) < 0) {
		fprintf(stderr, "Can not configure this PCM device.\n");
		snd_pcm_close(pcm_handle);
		return(NULL);
	}

	if (snd_pcm_hw_params_set_access(pcm_handle, hwparams, SND_PCM_ACCESS_RW_INTERLEAVED) < 0) {
		fprintf(stderr, "Error setting access.\n");
		snd_pcm_close(pcm_handle);
		return(NULL);
	}

	/* Set sample format */
	if (snd_pcm_hw_params_set_format(pcm_handle, hwparams, format) < 0) {
		fprintf(stderr, "Error setting format.\n");
		snd_pcm_close(pcm_handle);
		return(NULL);
	}

	/* Set sample rate. If the exact rate is not supported */
	/* by the hardware, use nearest possible rate.         */ 
	exact_rate = rate;
	if (snd_pcm_hw_params_set_rate_near(pcm_handle, hwparams, &exact_rate, 0) < 0) {
		fprintf(stderr, "Error setting rate.\n");
		snd_pcm_close(pcm_handle);
		return(NULL);
	}
	if (rate != exact_rate) {
        	fprintf(stderr, "The rate %d Hz is not supported by your hardware.\nUsing %d Hz instead.\n", rate, exact_rate);
	}

	/* Set number of channels */
	if (snd_pcm_hw_params_set_channels(pcm_handle, hwparams, channels) < 0) {
		fprintf(stderr, "Error setting channels.\n");
		snd_pcm_close(pcm_handle);
		return(NULL);
	}

	snd_pcm_hw_params_get_buffer_time_max(hwparams, &buffer_time, 0);
	if (buffer_time > 500000) buffer_time = 500000;

	if (buffer_time > 0) period_time = buffer_time / 4;
	else period_frames = buffer_frames / 4;

	if (period_time > 0) snd_pcm_hw_params_set_period_time_near(pcm_handle, hwparams, &period_time, 0);
	else snd_pcm_hw_params_set_period_size_near(pcm_handle, hwparams, &period_frames, 0);

	if (buffer_time > 0) snd_pcm_hw_params_set_buffer_time_near(pcm_handle, hwparams, &buffer_time, 0);
	else snd_pcm_hw_params_set_buffer_size_near(pcm_handle, hwparams, &buffer_frames);
						
	/* Apply HW parameter settings to */
	/* PCM device and prepare device  */
	if (snd_pcm_hw_params(pcm_handle, hwparams) < 0) {
		fprintf(stderr, "Error setting HW params.\n");
		snd_pcm_close(pcm_handle);
		return(NULL);
	}

	err = snd_pcm_prepare(pcm_handle);
	if (err < 0) {
		printf("Prepare error: %s\n", snd_strerror(err));
		snd_pcm_close(pcm_handle);
		return(NULL);
	}

	return pcm_handle;
}


int xrun_recovery(snd_pcm_t *handle, int err) {

	if (err == -EPIPE) {    /* under-run */
		err = snd_pcm_prepare(handle);
		if (err < 0) {
			printf("Can't recovery from underrun, prepare failed: %s\n", snd_strerror(err));
		}
		return 0;
	} else if (err == -ESTRPIPE) {
		while ((err = snd_pcm_resume(handle)) == -EAGAIN) {
			sleep(1);       /* wait until the suspend flag is released */
		}
		if (err < 0) {
			err = snd_pcm_prepare(handle);
			if (err < 0) {
				printf("Can't recovery from suspend, prepare failed: %s\n", snd_strerror(err));
			}
		}
		return(0);
	}
	return(err);
}

int alsa_play(void *handle, unsigned const char *data, int size, GYACHI_FORMAT_TYPE format_type) {
	snd_pcm_t *pcm_handle = handle;
	snd_pcm_format_t format = alsa_format(format_type);
	int channels = 1;
	int frame_bytes = (snd_pcm_format_width(format) / 8) * channels;
	int frame_count = size / frame_bytes;
	long result;

	if (!pcm_handle) {
		return(0);
	}

	do {
		result = snd_pcm_writei(pcm_handle, data, frame_count);
		if (result == -EAGAIN) {
			result = 0;
		}
		if (result < 0) {
			if (xrun_recovery(handle, result) < 0) {
				printf("Write error: %s\n", snd_strerror(result));
				result = -1;
				break;  /* skip one period */
			}
			result = 0;
			continue;  /* retry period */
		}

		data    += result * frame_bytes;
		frame_count  -= result;
	} while ((frame_count > 0) && (result >= 0));

	return(result);
}


int alsa_record(void *handle, unsigned char *data, int size, GYACHI_FORMAT_TYPE format_type) {
	snd_pcm_t *pcm_handle = handle;
	snd_pcm_format_t format = alsa_format(format_type);
	int channels = 1;
	int frame_bytes = (snd_pcm_format_width(format) / 8) * channels;
	int frame_count = size / frame_bytes;
	long result;

	if (!pcm_handle) {
		return(0);
	}

	do {
		result = snd_pcm_readi(pcm_handle, data, frame_count);
		if (result > 0) {
			data        += result * frame_bytes;
			frame_count -= result;
		}
	} while (result >= 1 && frame_count > 0);

	return(result);
}


int alsa_drain_device(void *handle)
{
	snd_pcm_t *pcm_handle = handle;

	if (!pcm_handle) {
		return(0);
	}

	/* Finish playing pending frames */ 
	snd_pcm_drain(pcm_handle);
	return(0);
}


int alsa_close_device(void *handle)
{
	snd_pcm_t *pcm_handle = handle;

	if (!pcm_handle) {
		return(0);
	}

	snd_pcm_close(pcm_handle);
	return(0);
}


GYACHI_SOUND_PLUGIN alsa_sound_plugin = {
	.name   = "ALSA",
	.description = NULL,
	.open   = alsa_open_device,
	.play   = alsa_play,
	.record = alsa_record,
	.drain  = alsa_drain_device,
	.close  = alsa_close_device
};

int alsa_plugin_init() {
	char description[512];

	sprintf (description, "ALSA plugin %s [gyachialsa.so]", snd_asoundlib_version());
	alsa_sound_plugin.description = strdup(description);
	register_sound_device(&alsa_sound_plugin);
	return(1);
}

PLUGIN_INFO plugin_info = {
	.type = PLUGIN_SOUND,
	.module_name = "GyachI-sound-plugin-ALSA",
	.description = "A plugin to use the ALSA subsystem for playing/recording sound", 
	.version     = "0.1", 
	.date        = "02/07/2008",
	.credits     = "Gregory D Hosler [ghosler ('at') users.sourceforge.net]",
	.sys_req     = "",
	.init        = alsa_plugin_init
};
