/*
 * Contributed by Stephane Eranian <eranian@gmail.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
 * CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * This file is part of libpfm, a performance monitoring support library for
 * applications on Linux.
 *
 * PMU: adl_glc (Alderlake GoldenCove P-Core)
 * Based on Intel JSON event table version   : 1.24
 * Based on Intel JSON event table published : 12/04/2023
 */

static const intel_x86_umask_t adl_glc_arith[]={
  { .uname   = "DIVIDER_ACTIVE",
    .udesc   = "This event is deprecated. Refer to new event ARITH.DIV_ACTIVE",
    .ucode   = 0x0900ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "DIV_ACTIVE",
    .udesc   = "Cycles when divide unit is busy executing divide or square root operations.",
    .ucode   = 0x0900ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "FPDIV_ACTIVE",
    .udesc   = "TBD",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "FP_DIVIDER_ACTIVE",
    .udesc   = "This event is deprecated. Refer to new event ARITH.FPDIV_ACTIVE",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "IDIV_ACTIVE",
    .udesc   = "This event counts the cycles the integer divider is busy.",
    .ucode   = 0x0800ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "INT_DIVIDER_ACTIVE",
    .udesc   = "This event is deprecated. Refer to new event ARITH.IDIV_ACTIVE",
    .ucode   = 0x0800ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
};

static const intel_x86_umask_t adl_glc_assists[]={
  { .uname   = "ANY",
    .udesc   = "Number of occurrences where a microcode assist is invoked by hardware.",
    .ucode   = 0x1b00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FP",
    .udesc   = "Counts all microcode FP assists.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HARDWARE",
    .udesc   = "TBD",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PAGE_FAULT",
    .udesc   = "TBD",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SSE_AVX_MIX",
    .udesc   = "TBD",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_baclears[]={
  { .uname   = "ANY",
    .udesc   = "Clears due to Unknown Branches.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_glc_br_inst_retired[]={
  { .uname   = "ALL_BRANCHES",
    .udesc   = "All branch instructions retired.",
    .ucode   = 0x0000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "COND",
    .udesc   = "Conditional branch instructions retired.",
    .ucode   = 0x1100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "COND_NTAKEN",
    .udesc   = "Not taken branch instructions retired.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "COND_TAKEN",
    .udesc   = "Taken conditional branch instructions retired.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "FAR_BRANCH",
    .udesc   = "Far branch instructions retired.",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "INDIRECT",
    .udesc   = "Indirect near branch instructions retired (excluding returns)",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "NEAR_CALL",
    .udesc   = "Direct and indirect near call instructions retired.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "NEAR_RETURN",
    .udesc   = "Return instructions retired.",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "NEAR_TAKEN",
    .udesc   = "Taken branch instructions retired.",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t adl_glc_br_misp_retired[]={
  { .uname   = "ALL_BRANCHES",
    .udesc   = "All mispredicted branch instructions retired.",
    .ucode   = 0x0000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "COND",
    .udesc   = "Mispredicted conditional branch instructions retired.",
    .ucode   = 0x1100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "COND_NTAKEN",
    .udesc   = "Mispredicted non-taken conditional branch instructions retired.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "COND_TAKEN",
    .udesc   = "number of branch instructions retired that were mispredicted and taken.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "INDIRECT",
    .udesc   = "Miss-predicted near indirect branch instructions retired (excluding returns)",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "INDIRECT_CALL",
    .udesc   = "Mispredicted indirect CALL retired.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "NEAR_TAKEN",
    .udesc   = "Number of near branch instructions retired that were mispredicted and taken.",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "RET",
    .udesc   = "This event counts the number of mispredicted ret instructions retired. Non PEBS",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t adl_glc_core_power[]={
  { .uname   = "LICENSE_1",
    .udesc   = "TBD",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "LICENSE_2",
    .udesc   = "TBD",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "LICENSE_3",
    .udesc   = "TBD",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_cpu_clk_unhalted[]={
  { .uname   = "C01",
    .udesc   = "Core clocks when the thread is in the C0.1 light-weight slower wakeup time but more power saving optimized state.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "C02",
    .udesc   = "Core clocks when the thread is in the C0.2 light-weight faster wakeup time but less power saving optimized state.",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "C0_WAIT",
    .udesc   = "Core clocks when the thread is in the C0.1 or C0.2 or running a PAUSE in C0 ACPI state.",
    .ucode   = 0x7000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DISTRIBUTED",
    .udesc   = "Cycle counts are evenly distributed between active threads in the Core.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ONE_THREAD_ACTIVE",
    .udesc   = "Core crystal clock cycles when this thread is unhalted and the other thread is halted.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PAUSE",
    .udesc   = "TBD",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PAUSE_INST",
    .udesc   = "TBD",
    .ucode   = 0x4000ull | (0x1 << INTEL_X86_CMASK_BIT) | (0x1 << INTEL_X86_EDGE_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C | _INTEL_X86_ATTR_E,
  },
  { .uname   = "REF_DISTRIBUTED",
    .udesc   = "Core crystal clock cycles. Cycle counts are evenly distributed between active threads in the Core.",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "REF_TSC",
    .udesc   = "Reference cycles when the core is not in halt state.",
    .ucode   = 0x0300ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "REF_TSC_P",
    .udesc   = "Reference cycles when the core is not in halt state.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "THREAD",
    .udesc   = "Core cycles when the thread is not in halt state",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "THREAD_P",
    .udesc   = "Thread cycles when thread is not in halt state",
    .ucode   = 0x0000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_cycle_activity[]={
  { .uname   = "CYCLES_L1D_MISS",
    .udesc   = "Cycles while L1 cache miss demand load is outstanding.",
    .ucode   = 0x0800ull | (0x8 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_L2_MISS",
    .udesc   = "Cycles while L2 cache miss demand load is outstanding.",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_MEM_ANY",
    .udesc   = "Cycles while memory subsystem has an outstanding load.",
    .ucode   = 0x1000ull | (0x10 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "STALLS_L1D_MISS",
    .udesc   = "Execution stalls while L1 cache miss demand load is outstanding.",
    .ucode   = 0x0c00ull | (0xc << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "STALLS_L2_MISS",
    .udesc   = "Execution stalls while L2 cache miss demand load is outstanding.",
    .ucode   = 0x0500ull | (0x5 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "STALLS_L3_MISS",
    .udesc   = "Execution stalls while L3 cache miss demand load is outstanding.",
    .ucode   = 0x0600ull | (0x6 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "STALLS_TOTAL",
    .udesc   = "Total execution stalls.",
    .ucode   = 0x0400ull | (0x4 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
};

static const intel_x86_umask_t adl_glc_decode[]={
  { .uname   = "LCP",
    .udesc   = "Stalls caused by changing prefix length of the instruction.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MS_BUSY",
    .udesc   = "Cycles the Microcode Sequencer is busy.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_dsb2mite_switches[]={
  { .uname   = "PENALTY_CYCLES",
    .udesc   = "DSB-to-MITE switch true penalty cycles.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_glc_dtlb_load_misses[]={
  { .uname   = "STLB_HIT",
    .udesc   = "Loads that miss the DTLB and hit the STLB.",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_ACTIVE",
    .udesc   = "Cycles when at least one PMH is busy with a page walk for a demand load.",
    .ucode   = 0x1000ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "WALK_COMPLETED",
    .udesc   = "Load miss in all TLB levels causes a page walk that completes. (All page sizes)",
    .ucode   = 0x0e00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_COMPLETED_1G",
    .udesc   = "Page walks completed due to a demand data load to a 1G page.",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_COMPLETED_2M_4M",
    .udesc   = "Page walks completed due to a demand data load to a 2M/4M page.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_COMPLETED_4K",
    .udesc   = "Page walks completed due to a demand data load to a 4K page.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_PENDING",
    .udesc   = "Number of page walks outstanding for a demand load in the PMH each cycle.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_dtlb_store_misses[]={
  { .uname   = "STLB_HIT",
    .udesc   = "Stores that miss the DTLB and hit the STLB.",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_ACTIVE",
    .udesc   = "Cycles when at least one PMH is busy with a page walk for a store.",
    .ucode   = 0x1000ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "WALK_COMPLETED",
    .udesc   = "Store misses in all TLB levels causes a page walk that completes. (All page sizes)",
    .ucode   = 0x0e00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_COMPLETED_1G",
    .udesc   = "Page walks completed due to a demand data store to a 1G page.",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_COMPLETED_2M_4M",
    .udesc   = "Page walks completed due to a demand data store to a 2M/4M page.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_COMPLETED_4K",
    .udesc   = "Page walks completed due to a demand data store to a 4K page.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_PENDING",
    .udesc   = "Number of page walks outstanding for a store in the PMH each cycle.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_exe_activity[]={
  { .uname   = "1_PORTS_UTIL",
    .udesc   = "Cycles total of 1 uop is executed on all ports and Reservation Station was not empty.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "2_PORTS_UTIL",
    .udesc   = "Cycles total of 2 uops are executed on all ports and Reservation Station was not empty.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "3_PORTS_UTIL",
    .udesc   = "Cycles total of 3 uops are executed on all ports and Reservation Station was not empty.",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "4_PORTS_UTIL",
    .udesc   = "Cycles total of 4 uops are executed on all ports and Reservation Station was not empty.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "BOUND_ON_LOADS",
    .udesc   = "Execution stalls while memory subsystem has an outstanding load.",
    .ucode   = 0x2100ull | (0x5 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "BOUND_ON_STORES",
    .udesc   = "Cycles where the Store Buffer was full and no loads caused an execution stall.",
    .ucode   = 0x4000ull | (0x2 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "EXE_BOUND_0_PORTS",
    .udesc   = "Cycles no uop executed while RS was not empty, the SB was not full and there was no outstanding load.",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_fp_arith_dispatched[]={
  { .uname   = "PORT_0",
    .udesc   = "TBD",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PORT_1",
    .udesc   = "TBD",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PORT_5",
    .udesc   = "TBD",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "V0",
    .udesc   = "TBD",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "V1",
    .udesc   = "TBD",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "V5",
    .udesc   = "TBD",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_fp_arith_inst_retired[]={
  { .uname   = "128B_PACKED_DOUBLE",
    .udesc   = "Counts number of SSE/AVX computational 128-bit packed double precision floating-point instructions retired; some instructions will count twice as noted below.  Each count represents 2 computation operations, one for each element.  Applies to SSE* and AVX* packed double precision floating-point instructions: ADD SUB HADD HSUB SUBADD MUL DIV MIN MAX SQRT DPP FM(N)ADD/SUB.  DPP and FM(N)ADD/SUB instructions count twice as they perform 2 calculations per element.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "128B_PACKED_SINGLE",
    .udesc   = "Counts number of SSE/AVX computational 128-bit packed single precision floating-point instructions retired; some instructions will count twice as noted below.  Each count represents 4 computation operations, one for each element.  Applies to SSE* and AVX* packed single precision floating-point instructions: ADD SUB MUL DIV MIN MAX RCP14 RSQRT14 SQRT DPP FM(N)ADD/SUB. DPP and FM(N)ADD/SUB instructions count twice as they perform 2 calculations per element.",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "256B_PACKED_DOUBLE",
    .udesc   = "Counts number of SSE/AVX computational 256-bit packed double precision floating-point instructions retired; some instructions will count twice as noted below.  Each count represents 4 computation operations, one for each element.  Applies to SSE* and AVX* packed double precision floating-point instructions: ADD SUB HADD HSUB SUBADD MUL DIV MIN MAX SQRT FM(N)ADD/SUB. FM(N)ADD/SUB instructions count twice as they perform 2 calculations per element.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "256B_PACKED_SINGLE",
    .udesc   = "Counts number of SSE/AVX computational 256-bit packed single precision floating-point instructions retired; some instructions will count twice as noted below.  Each count represents 8 computation operations, one for each element.  Applies to SSE* and AVX* packed single precision floating-point instructions: ADD SUB HADD HSUB SUBADD MUL DIV MIN MAX SQRT RSQRT RCP DPP FM(N)ADD/SUB. DPP and FM(N)ADD/SUB instructions count twice as they perform 2 calculations per element.",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "4_FLOPS",
    .udesc   = "Counts number of SSE/AVX computational 128-bit packed single and 256-bit packed double precision FP instructions retired; some instructions will count twice as noted below.  Each count represents 2 or/and 4 computation operations, 1 for each element.  Applies to SSE* and AVX* packed single precision and packed double precision FP instructions: ADD SUB HADD HSUB SUBADD MUL DIV MIN MAX RCP14 RSQRT14 SQRT DPP FM(N)ADD/SUB. DPP and FM(N)ADD/SUB count twice as they perform 2 calculations per element.",
    .ucode   = 0x1800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SCALAR",
    .udesc   = "Counts number of SSE/AVX computational scalar floating-point instructions retired; some instructions will count twice as noted below.  Applies to SSE* and AVX* scalar, double and single precision floating-point: ADD SUB MUL DIV MIN MAX RCP14 RSQRT14 RANGE SQRT DPP FM(N)ADD/SUB. DPP and FM(N)ADD/SUB instructions count twice as they perform multiple calculations per element.",
    .ucode   = 0x0300ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SCALAR_DOUBLE",
    .udesc   = "Counts number of SSE/AVX computational scalar double precision floating-point instructions retired; some instructions will count twice as noted below.  Each count represents 1 computational operation. Applies to SSE* and AVX* scalar double precision floating-point instructions: ADD SUB MUL DIV MIN MAX SQRT FM(N)ADD/SUB. FM(N)ADD/SUB instructions count twice as they perform 2 calculations per element.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SCALAR_SINGLE",
    .udesc   = "Counts number of SSE/AVX computational scalar single precision floating-point instructions retired; some instructions will count twice as noted below.  Each count represents 1 computational operation. Applies to SSE* and AVX* scalar single precision floating-point instructions: ADD SUB MUL DIV MIN MAX SQRT RSQRT RCP FM(N)ADD/SUB. FM(N)ADD/SUB instructions count twice as they perform 2 calculations per element.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "VECTOR",
    .udesc   = "Number of any Vector retired FP arithmetic instructions",
    .ucode   = 0xfc00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_frontend_retired[]={
  { .uname   = "ANY_DSB_MISS",
    .udesc   = "Retired Instructions who experienced DSB miss.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "DSB_MISS",
    .udesc   = "Retired Instructions who experienced a critical DSB miss.",
    .ucode   = 0x1100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "ITLB_MISS",
    .udesc   = "Retired Instructions who experienced iTLB true miss.",
    .ucode   = 0x1400ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "L1I_MISS",
    .udesc   = "Retired Instructions who experienced Instruction L1 Cache true miss.",
    .ucode   = 0x1200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "L2_MISS",
    .udesc   = "Retired Instructions who experienced Instruction L2 Cache true miss.",
    .ucode   = 0x1300ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "LATENCY_GE_1",
    .udesc   = "Retired instructions after front-end starvation of at least 1 cycle",
    .ucode   = 0x60010600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
  },
  { .uname   = "LATENCY_GE_128",
    .udesc   = "Retired instructions that are fetched after an interval where the front-end delivered no uops for a period of 128 cycles which was not interrupted by a back-end stall.",
    .ucode   = 0x60800600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
  },
  { .uname   = "LATENCY_GE_16",
    .udesc   = "Retired instructions that are fetched after an interval where the front-end delivered no uops for a period of 16 cycles which was not interrupted by a back-end stall.",
    .ucode   = 0x60100600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
  },
  { .uname   = "LATENCY_GE_2",
    .udesc   = "Retired instructions after front-end starvation of at least 2 cycles",
    .ucode   = 0x60020600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
  },
  { .uname   = "LATENCY_GE_256",
    .udesc   = "Retired instructions that are fetched after an interval where the front-end delivered no uops for a period of 256 cycles which was not interrupted by a back-end stall.",
    .ucode   = 0x61000600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
  },
  { .uname   = "LATENCY_GE_2_BUBBLES_GE_1",
    .udesc   = "Retired instructions that are fetched after an interval where the front-end had at least 1 bubble-slot for a period of 2 cycles which was not interrupted by a back-end stall.",
    .ucode   = 0x10020600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "LATENCY_GE_32",
    .udesc   = "Retired instructions that are fetched after an interval where the front-end delivered no uops for a period of 32 cycles which was not interrupted by a back-end stall.",
    .ucode   = 0x60200600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
  },
  { .uname   = "LATENCY_GE_4",
    .udesc   = "Retired instructions that are fetched after an interval where the front-end delivered no uops for a period of 4 cycles which was not interrupted by a back-end stall.",
    .ucode   = 0x60040600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
  },
  { .uname   = "LATENCY_GE_512",
    .udesc   = "Retired instructions that are fetched after an interval where the front-end delivered no uops for a period of 512 cycles which was not interrupted by a back-end stall.",
    .ucode   = 0x62000600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
  },
  { .uname   = "LATENCY_GE_64",
    .udesc   = "Retired instructions that are fetched after an interval where the front-end delivered no uops for a period of 64 cycles which was not interrupted by a back-end stall.",
    .ucode   = 0x60400600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
  },
  { .uname   = "LATENCY_GE_8",
    .udesc   = "Retired instructions that are fetched after an interval where the front-end delivered no uops for a period of 8 cycles which was not interrupted by a back-end stall.",
    .ucode   = 0x60080600ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_FETHR | INTEL_X86_PEBS,
  },
  { .uname   = "MS_FLOWS",
    .udesc   = "TBD",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "STLB_MISS",
    .udesc   = "Retired Instructions who experienced STLB (2nd level TLB) true miss.",
    .ucode   = 0x1500ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "UNKNOWN_BRANCH",
    .udesc   = "TBD",
    .ucode   = 0x1700ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t adl_glc_icache_data[]={
  { .uname   = "STALLS",
    .udesc   = "Cycles where a code fetch is stalled due to L1 instruction cache miss.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_glc_icache_tag[]={
  { .uname   = "STALLS",
    .udesc   = "Cycles where a code fetch is stalled due to L1 instruction cache tag miss.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_glc_idq[]={
  { .uname   = "DSB_CYCLES_ANY",
    .udesc   = "Cycles Decode Stream Buffer (DSB) is delivering any Uop",
    .ucode   = 0x0800ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "DSB_CYCLES_OK",
    .udesc   = "Cycles DSB is delivering optimal number of Uops",
    .ucode   = 0x0800ull | (0x6 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "DSB_UOPS",
    .udesc   = "Uops delivered to Instruction Decode Queue (IDQ) from the Decode Stream Buffer (DSB) path",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MITE_CYCLES_ANY",
    .udesc   = "Cycles MITE is delivering any Uop",
    .ucode   = 0x0400ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "MITE_CYCLES_OK",
    .udesc   = "Cycles MITE is delivering optimal number of Uops",
    .ucode   = 0x0400ull | (0x6 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "MITE_UOPS",
    .udesc   = "Uops delivered to Instruction Decode Queue (IDQ) from MITE path",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MS_CYCLES_ANY",
    .udesc   = "Cycles when uops are being delivered to IDQ while MS is busy",
    .ucode   = 0x2000ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "MS_SWITCHES",
    .udesc   = "Number of switches from DSB or MITE to the MS",
    .ucode   = 0x2000ull | (0x1 << INTEL_X86_CMASK_BIT) | (0x1 << INTEL_X86_EDGE_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C | _INTEL_X86_ATTR_E,
  },
  { .uname   = "MS_UOPS",
    .udesc   = "Uops delivered to IDQ while MS is busy",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_idq_uops_not_delivered[]={
  { .uname   = "CORE",
    .udesc   = "Uops not delivered by IDQ when backend of the machine is not stalled [This event is alias to IDQ_BUBBLES.CORE]",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "CYCLES_0_UOPS_DELIV_CORE",
    .udesc   = "Cycles when no uops are not delivered by the IDQ when backend of the machine is not stalled [This event is alias to IDQ_BUBBLES.CYCLES_0_UOPS_DELIV.CORE]",
    .ucode   = 0x0100ull | (0x6 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_FE_WAS_OK",
    .udesc   = "Cycles when optimal number of uops was delivered to the back-end when the back-end is not stalled [This event is alias to IDQ_BUBBLES.CYCLES_FE_WAS_OK]",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_INV_BIT) | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_I | _INTEL_X86_ATTR_C,
  },
};

static const intel_x86_umask_t adl_glc_inst_decoded[]={
  { .uname   = "DECODERS",
    .udesc   = "Instruction decoders utilized in a cycle",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_glc_inst_retired[]={
  { .uname   = "ANY",
    .udesc   = "Number of instructions retired. Fixed Counter - architectural event",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "ANY_P",
    .udesc   = "Number of instructions retired. General Counter - architectural event",
    .ucode   = 0x0000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "MACRO_FUSED",
    .udesc   = "TBD",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "NOP",
    .udesc   = "Retired NOP instructions.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "PREC_DIST",
    .udesc   = "Precise instruction retired with PEBS precise-distribution",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "REP_ITERATION",
    .udesc   = "Iterations of Repeat string retired instructions.",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t adl_glc_int_misc[]={
  { .uname   = "CLEARS_COUNT",
    .udesc   = "Clears speculative count",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_CMASK_BIT) | (0x1 << INTEL_X86_EDGE_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C | _INTEL_X86_ATTR_E,
  },
  { .uname   = "CLEAR_RESTEER_CYCLES",
    .udesc   = "Counts cycles after recovery from a branch misprediction or machine clear till the first uop is issued from the resteered path.",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RECOVERY_CYCLES",
    .udesc   = "Core cycles the allocator was stalled due to recovery from earlier clear event for this thread",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "UNKNOWN_BRANCH_CYCLES",
    .udesc   = "Bubble cycles of BAClear (Unknown Branch).",
    .ucode   = 0x0700ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "UOP_DROPPING",
    .udesc   = "TMA slots where uops got dropped",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_int_vec_retired[]={
  { .uname   = "128BIT",
    .udesc   = "TBD",
    .ucode   = 0x1300ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "256BIT",
    .udesc   = "TBD",
    .ucode   = 0xac00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ADD_128",
    .udesc   = "integer ADD, SUB, SAD 128-bit vector instructions.",
    .ucode   = 0x0300ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ADD_256",
    .udesc   = "integer ADD, SUB, SAD 256-bit vector instructions.",
    .ucode   = 0x0c00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MUL_256",
    .udesc   = "TBD",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SHUFFLES",
    .udesc   = "TBD",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "VNNI_128",
    .udesc   = "TBD",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "VNNI_256",
    .udesc   = "TBD",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_itlb_misses[]={
  { .uname   = "STLB_HIT",
    .udesc   = "Instruction fetch requests that miss the ITLB and hit the STLB.",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_ACTIVE",
    .udesc   = "Cycles when at least one PMH is busy with a page walk for code (instruction fetch) request.",
    .ucode   = 0x1000ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "WALK_COMPLETED",
    .udesc   = "Code miss in all TLB levels causes a page walk that completes. (All page sizes)",
    .ucode   = 0x0e00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_COMPLETED_2M_4M",
    .udesc   = "Code miss in all TLB levels causes a page walk that completes. (2M/4M)",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_COMPLETED_4K",
    .udesc   = "Code miss in all TLB levels causes a page walk that completes. (4K)",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "WALK_PENDING",
    .udesc   = "Number of page walks outstanding for an outstanding code request in the PMH each cycle.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_l1d[]={
  { .uname   = "HWPF_MISS",
    .udesc   = "TBD",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "REPLACEMENT",
    .udesc   = "Counts the number of cache lines replaced in L1 data cache.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_l1d_pend_miss[]={
  { .uname   = "FB_FULL",
    .udesc   = "Number of cycles a demand request has waited due to L1D Fill Buffer (FB) unavailability.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "FB_FULL_PERIODS",
    .udesc   = "Number of phases a demand request has waited due to L1D Fill Buffer (FB) unavailability.",
    .ucode   = 0x0200ull | (0x1 << INTEL_X86_CMASK_BIT) | (0x1 << INTEL_X86_EDGE_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C | _INTEL_X86_ATTR_E,
  },
  { .uname   = "L2_STALL",
    .udesc   = "This event is deprecated. Refer to new event L1D_PEND_MISS.L2_STALLS",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "L2_STALLS",
    .udesc   = "Number of cycles a demand request has waited due to L1D due to lack of L2 resources.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PENDING",
    .udesc   = "Number of L1D misses that are outstanding",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PENDING_CYCLES",
    .udesc   = "Cycles with L1D load Misses outstanding.",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
};

static const intel_x86_umask_t adl_glc_l2_lines_in[]={
  { .uname   = "ALL",
    .udesc   = "L2 cache lines filling L2",
    .ucode   = 0x1f00ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_glc_l2_lines_out[]={
  { .uname   = "USELESS_HWPF",
    .udesc   = "Cache lines that have been L2 hardware prefetched but not used by demand accesses",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_glc_l2_rqsts[]={
  { .uname   = "ALL_CODE_RD",
    .udesc   = "L2 code requests",
    .ucode   = 0xe400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ALL_DEMAND_DATA_RD",
    .udesc   = "Demand Data Read access L2 cache",
    .ucode   = 0xe100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ALL_DEMAND_MISS",
    .udesc   = "Demand requests that miss L2 cache",
    .ucode   = 0x2700ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ALL_HWPF",
    .udesc   = "TBD",
    .ucode   = 0xf000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "ALL_RFO",
    .udesc   = "RFO requests to L2 cache.",
    .ucode   = 0xe200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "CODE_RD_HIT",
    .udesc   = "L2 cache hits when fetching instructions, code reads.",
    .ucode   = 0xc400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "CODE_RD_MISS",
    .udesc   = "L2 cache misses when fetching instructions",
    .ucode   = 0x2400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_HIT",
    .udesc   = "Demand Data Read requests that hit L2 cache",
    .ucode   = 0xc100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_MISS",
    .udesc   = "Demand Data Read miss L2 cache",
    .ucode   = 0x2100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "HWPF_MISS",
    .udesc   = "TBD",
    .ucode   = 0x3000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MISS",
    .udesc   = "Read requests with true-miss in L2 cache. [This event is alias to L2_REQUEST.MISS]",
    .ucode   = 0x3f00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "REFERENCES",
    .udesc   = "All accesses to L2 cache [This event is alias to L2_REQUEST.ALL]",
    .ucode   = 0xff00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RFO_HIT",
    .udesc   = "RFO requests that hit L2 cache.",
    .ucode   = 0xc200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RFO_MISS",
    .udesc   = "RFO requests that miss L2 cache",
    .ucode   = 0x2200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SWPF_HIT",
    .udesc   = "SW prefetch requests that hit L2 cache.",
    .ucode   = 0xc800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SWPF_MISS",
    .udesc   = "SW prefetch requests that miss L2 cache.",
    .ucode   = 0x2800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_ld_blocks[]={
  { .uname   = "ADDRESS_ALIAS",
    .udesc   = "False dependencies in MOB due to partial compare on address.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "NO_SR",
    .udesc   = "The number of times that split load operations are temporarily blocked because all resources for handling the split accesses are in use.",
    .ucode   = 0x8800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "STORE_FORWARD",
    .udesc   = "Loads blocked due to overlapping with a preceding store that cannot be forwarded.",
    .ucode   = 0x8200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_load_hit_prefetch[]={
  { .uname   = "SWPF",
    .udesc   = "Counts the number of demand load dispatches that hit L1D fill buffer (FB) allocated for software prefetch.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_glc_longest_lat_cache[]={
  { .uname   = "MISS",
    .udesc   = "Core-originated cacheable requests that missed L3  (Except hardware prefetches to the L3)",
    .ucode   = 0x4100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "REFERENCE",
    .udesc   = "Core-originated cacheable requests that refer to L3 (Except hardware prefetches to the L3)",
    .ucode   = 0x4f00ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_lsd[]={
  { .uname   = "CYCLES_ACTIVE",
    .udesc   = "Cycles Uops delivered by the LSD, but didn't come from the decoder.",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_OK",
    .udesc   = "Cycles optimal number of Uops delivered by the LSD, but did not come from the decoder.",
    .ucode   = 0x0100ull | (0x6 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "UOPS",
    .udesc   = "Number of Uops delivered by the LSD.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_machine_clears[]={
  { .uname   = "COUNT",
    .udesc   = "Number of machine clears (nukes) of any type.",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_CMASK_BIT) | (0x1 << INTEL_X86_EDGE_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C | _INTEL_X86_ATTR_E,
  },
  { .uname   = "MEMORY_ORDERING",
    .udesc   = "Number of machine clears due to memory ordering conflicts.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SMC",
    .udesc   = "Self-modifying code (SMC) detected.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_memory_activity[]={
  { .uname   = "CYCLES_L1D_MISS",
    .udesc   = "Cycles while L1 cache miss demand load is outstanding.",
    .ucode   = 0x0200ull | (0x2 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "STALLS_L1D_MISS",
    .udesc   = "Execution stalls while L1 cache miss demand load is outstanding.",
    .ucode   = 0x0300ull | (0x3 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "STALLS_L2_MISS",
    .udesc   = "Execution stalls while L2 cache miss demand cacheable load request is outstanding.",
    .ucode   = 0x0500ull | (0x5 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "STALLS_L3_MISS",
    .udesc   = "Execution stalls while L3 cache miss demand cacheable load request is outstanding.",
    .ucode   = 0x0900ull | (0x9 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
};

static const intel_x86_umask_t adl_glc_mem_inst_retired[]={
  { .uname   = "ALL_LOADS",
    .udesc   = "Retired load instructions.",
    .ucode   = 0x8100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "ALL_STORES",
    .udesc   = "Retired store instructions.",
    .ucode   = 0x8200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "ANY",
    .udesc   = "All retired memory instructions.",
    .ucode   = 0x8300ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "LOCK_LOADS",
    .udesc   = "Retired load instructions with locked access.",
    .ucode   = 0x2100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "SPLIT_LOADS",
    .udesc   = "Retired load instructions that split across a cacheline boundary.",
    .ucode   = 0x4100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "SPLIT_STORES",
    .udesc   = "Retired store instructions that split across a cacheline boundary.",
    .ucode   = 0x4200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "STLB_MISS_LOADS",
    .udesc   = "Retired load instructions that miss the STLB.",
    .ucode   = 0x1100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "STLB_MISS_STORES",
    .udesc   = "Retired store instructions that miss the STLB.",
    .ucode   = 0x1200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t adl_glc_mem_load_completed[]={
  { .uname   = "L1_MISS_ANY",
    .udesc   = "Completed demand load uops that miss the L1 d-cache.",
    .ucode   = 0xfd00ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_glc_mem_load_l3_hit_retired[]={
  { .uname   = "XSNP_FWD",
    .udesc   = "Retired load instructions whose data sources were HitM responses from shared L3",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "XSNP_HIT",
    .udesc   = "Retired load instructions whose data sources were L3 and cross-core snoop hits in on-pkg core cache",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "XSNP_HITM",
    .udesc   = "Retired load instructions whose data sources were HitM responses from shared L3",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "XSNP_MISS",
    .udesc   = "Retired load instructions whose data sources were L3 hit and cross-core snoop missed in on-pkg core cache.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "XSNP_NONE",
    .udesc   = "Retired load instructions whose data sources were hits in L3 without snoops required",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "XSNP_NO_FWD",
    .udesc   = "Retired load instructions whose data sources were L3 and cross-core snoop hits in on-pkg core cache",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t adl_glc_mem_load_l3_miss_retired[]={
  { .uname   = "LOCAL_DRAM",
    .udesc   = "Retired load instructions which data sources missed L3 but serviced from local dram",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_PEBS | INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_glc_mem_load_misc_retired[]={
  { .uname   = "UC",
    .udesc   = "Retired instructions with at least 1 uncacheable load or lock.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_PEBS | INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_glc_mem_load_retired[]={
  { .uname   = "FB_HIT",
    .udesc   = "Number of completed demand load requests that missed the L1, but hit the FB(fill buffer), because a preceding miss to the same cacheline initiated the line to be brought into L1, but data is not yet ready in L1.",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "L1_HIT",
    .udesc   = "Retired load instructions with L1 cache hits as data sources",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "L1_MISS",
    .udesc   = "Retired load instructions missed L1 cache as data sources",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "L2_HIT",
    .udesc   = "Retired load instructions with L2 cache hits as data sources",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "L2_MISS",
    .udesc   = "Retired load instructions missed L2 cache as data sources",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "L3_HIT",
    .udesc   = "Retired load instructions with L3 cache hits as data sources",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "L3_MISS",
    .udesc   = "Retired load instructions missed L3 cache as data sources",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t adl_glc_mem_store_retired[]={
  { .uname   = "L2_HIT",
    .udesc   = "TBD",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_glc_mem_trans_retired[]={
  { .uname  = "LOAD_LATENCY",
    .udesc  = "Memory load instructions retired above programmed clocks, minimum threshold value is 3 (Precise Event and ldlat required)",
    .ucode  = 0x100,
    .uflags = INTEL_X86_NCOMBO | INTEL_X86_PEBS | INTEL_X86_LDLAT | INTEL_X86_DFL,
  },
  { .uname   = "LOAD_LATENCY_GT_128",
    .udesc   = "Counts randomly selected loads when the latency from first dispatch to completion is greater than 128 cycles.",
    .uequiv  = "LOAD_LATENCY:ldlat=128",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS | INTEL_X86_LDLAT,
  },
  { .uname   = "LOAD_LATENCY_GT_16",
    .udesc   = "Counts randomly selected loads when the latency from first dispatch to completion is greater than 16 cycles.",
    .uequiv  = "LOAD_LATENCY:ldlat=16",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "LOAD_LATENCY_GT_256",
    .udesc   = "Counts randomly selected loads when the latency from first dispatch to completion is greater than 256 cycles.",
    .uequiv  = "LOAD_LATENCY:ldlat=256",
    .ucode   = 0x10000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "LOAD_LATENCY_GT_32",
    .udesc   = "Counts randomly selected loads when the latency from first dispatch to completion is greater than 32 cycles.",
    .uequiv  = "LOAD_LATENCY:ldlat=32",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "LOAD_LATENCY_GT_4",
    .udesc   = "Counts randomly selected loads when the latency from first dispatch to completion is greater than 4 cycles.",
    .uequiv  = "LOAD_LATENCY:ldlat=4",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "LOAD_LATENCY_GT_512",
    .udesc   = "Counts randomly selected loads when the latency from first dispatch to completion is greater than 512 cycles.",
    .ucode   = 0x20000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "LOAD_LATENCY_GT_64",
    .udesc   = "Counts randomly selected loads when the latency from first dispatch to completion is greater than 64 cycles.",
    .uequiv  = "LOAD_LATENCY:ldlat=64",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "LOAD_LATENCY_GT_8",
    .udesc   = "Counts randomly selected loads when the latency from first dispatch to completion is greater than 8 cycles.",
    .uequiv  = "LOAD_LATENCY:ldlat=8",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
  { .uname   = "STORE_SAMPLE",
    .udesc   = "Retired memory store access operations. A PDist event for PEBS Store Latency Facility.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_PEBS,
  },
};

static const intel_x86_umask_t adl_glc_mem_uop_retired[]={
  { .uname   = "ANY",
    .udesc   = "Retired memory uops for any access",
    .ucode   = 0x0300ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_glc_misc2_retired[]={
  { .uname   = "LFENCE",
    .udesc   = "LFENCE instructions retired",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_glc_misc_retired[]={
  { .uname   = "LBR_INSERTS",
    .udesc   = "Increments whenever there is an update to the LBR array.",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_glc_ocr[]={
  { .uname   = "DEMAND_DATA_RD_ANY_RESPONSE",
    .udesc   = "Counts demand data reads that have any type of response.",
    .ucode   = 0x1000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_DRAM",
    .udesc   = "Counts demand data reads that were supplied by DRAM.",
    .ucode   = 0x18400000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_L3_HIT_SNOOP_HITM",
    .udesc   = "Counts demand data reads that resulted in a snoop hit in another cores caches, data forwarding is required as the data is modified.",
    .ucode   = 0x10003c000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_L3_HIT_SNOOP_HIT_WITH_FWD",
    .udesc   = "Counts demand data reads that resulted in a snoop hit in another cores caches which forwarded the unmodified data to the requesting core.",
    .ucode   = 0x8003c000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD_L3_MISS",
    .udesc   = "Counts demand data reads that were not supplied by the L3 cache.",
    .ucode   = 0x3fbfc0000100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO_ANY_RESPONSE",
    .udesc   = "Counts demand read for ownership (RFO) requests and software prefetches for exclusive ownership (PREFETCHW) that have any type of response.",
    .ucode   = 0x1000200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO_L3_HIT_SNOOP_HITM",
    .udesc   = "Counts demand read for ownership (RFO) requests and software prefetches for exclusive ownership (PREFETCHW) that resulted in a snoop hit in another cores caches, data forwarding is required as the data is modified.",
    .ucode   = 0x10003c000200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_RFO_L3_MISS",
    .udesc   = "Counts demand read for ownership (RFO) requests and software prefetches for exclusive ownership (PREFETCHW) that were not supplied by the L3 cache.",
    .ucode   = 0x3fbfc0000200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "STREAMING_WR_ANY_RESPONSE",
    .udesc   = "Counts streaming stores that have any type of response.",
    .ucode   = 0x1080000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_offcore_requests[]={
  { .uname   = "ALL_REQUESTS",
    .udesc   = "TBD",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DATA_RD",
    .udesc   = "Demand and prefetch data reads",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD",
    .udesc   = "Demand Data Read requests sent to uncore",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "L3_MISS_DEMAND_DATA_RD",
    .udesc   = "Counts demand data read requests that miss the L3 cache.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_offcore_requests_outstanding[]={
  { .uname   = "ALL_DATA_RD",
    .udesc   = "This event is deprecated. Refer to new event OFFCORE_REQUESTS_OUTSTANDING.DATA_RD",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "CYCLES_WITH_DATA_RD",
    .udesc   = "TBD",
    .ucode   = 0x0800ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_WITH_DEMAND_DATA_RD",
    .udesc   = "Cycles where at least 1 outstanding demand data read request is pending.",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_WITH_DEMAND_RFO",
    .udesc   = "For every cycle where the core is waiting on at least 1 outstanding Demand RFO request, increments by 1.",
    .ucode   = 0x0400ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "DATA_RD",
    .udesc   = "TBD",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "DEMAND_DATA_RD",
    .udesc   = "For every cycle, increments by the number of outstanding demand data read requests pending.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "L3_MISS_DEMAND_DATA_RD",
    .udesc   = "For every cycle, increments by the number of demand data read requests pending that are known to have missed the L3 cache.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_resource_stalls[]={
  { .uname   = "SB",
    .udesc   = "Cycles stalled due to no store buffers available. (not including draining form sync).",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SCOREBOARD",
    .udesc   = "Counts cycles where the pipeline is stalled due to serializing operations.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_rs[]={
  { .uname   = "EMPTY",
    .udesc   = "Cycles when Reservation Station (RS) is empty for the thread.",
    .ucode   = 0x0700ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "EMPTY_COUNT",
    .udesc   = "Counts end of periods where the Reservation Station (RS) was empty.",
    .ucode   = 0x0700ull | (0x1 << INTEL_X86_INV_BIT) | (0x1 << INTEL_X86_CMASK_BIT) | (0x1 << INTEL_X86_EDGE_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_I | _INTEL_X86_ATTR_C | _INTEL_X86_ATTR_E,
  },
};

static const intel_x86_umask_t adl_glc_rs_empty[]={
  { .uname   = "COUNT",
    .udesc   = "This event is deprecated. Refer to new event RS.EMPTY_COUNT",
    .ucode   = 0x0700ull | (0x1 << INTEL_X86_INV_BIT) | (0x1 << INTEL_X86_CMASK_BIT) | (0x1 << INTEL_X86_EDGE_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_I | _INTEL_X86_ATTR_C | _INTEL_X86_ATTR_E,
  },
  { .uname   = "CYCLES",
    .udesc   = "This event is deprecated. Refer to new event RS.EMPTY",
    .ucode   = 0x0700ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_sq_misc[]={
  { .uname   = "BUS_LOCK",
    .udesc   = "Counts bus locks, accounts for cache line split locks and UC locks.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_glc_sw_prefetch_access[]={
  { .uname   = "NTA",
    .udesc   = "Number of PREFETCHNTA instructions executed.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PREFETCHW",
    .udesc   = "Number of PREFETCHW instructions executed.",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "T0",
    .udesc   = "Number of PREFETCHT0 instructions executed.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "T1_T2",
    .udesc   = "Number of PREFETCHT1 or PREFETCHT2 instructions executed.",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_topdown[]={
  { .uname   = "BACKEND_BOUND_SLOTS",
    .udesc   = "TMA slots where no uops were being issued due to lack of back-end resources (Topdown L1)",
    .ucode   = 0x8300ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "BAD_SPEC_SLOTS",
    .udesc   = "TMA slots wasted due to incorrect speculations (Topdown L1)",
    .ucode   = 0x8100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "BR_MISPREDICT_SLOTS",
    .udesc   = "TMA slots wasted due to incorrect speculation by branch mispredictions",
    .ucode   = 0x8500ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MEMORY_BOUND_SLOTS",
    .udesc   = "TMA slots wasted due to memory accesses (TopdownL2)",
    .ucode   = 0x8700ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "RETIRING_SLOTS",
    .udesc   = "TMA slots where instructions are retiring (Topdown L1)",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SLOTS",
    .udesc   = "TMA slots available for an unhalted logical processor. Fixed counter - architectural event",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SLOTS_P",
    .udesc   = "TMA slots available for an unhalted logical processor. General counter - architectural event",
    .ucode   = 0x01a4ull,
    .uflags  = INTEL_X86_NCOMBO | INTEL_X86_CODE_OVERRIDE,
  },
};

static const intel_x86_umask_t adl_glc_uops_decoded[]={
  { .uname   = "DEC0_UOPS",
    .udesc   = "TBD",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_glc_uops_dispatched[]={
  { .uname   = "PORT_0",
    .udesc   = "Uops executed on port 0",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PORT_1",
    .udesc   = "Uops executed on port 1",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PORT_2_3_10",
    .udesc   = "Uops executed on ports 2, 3 and 10",
    .ucode   = 0x0400ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PORT_4_9",
    .udesc   = "Uops executed on ports 4 and 9",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PORT_5_11",
    .udesc   = "Uops executed on ports 5 and 11",
    .ucode   = 0x2000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PORT_6",
    .udesc   = "Uops executed on port 6",
    .ucode   = 0x4000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "PORT_7_8",
    .udesc   = "Uops executed on ports 7 and 8",
    .ucode   = 0x8000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_uops_executed[]={
  { .uname   = "CORE_CYCLES_GE_1",
    .udesc   = "Cycles at least 1 micro-op is executed from any thread on physical core.",
    .ucode   = 0x0200ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CORE_CYCLES_GE_2",
    .udesc   = "Cycles at least 2 micro-op is executed from any thread on physical core.",
    .ucode   = 0x0200ull | (0x2 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CORE_CYCLES_GE_3",
    .udesc   = "Cycles at least 3 micro-op is executed from any thread on physical core.",
    .ucode   = 0x0200ull | (0x3 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CORE_CYCLES_GE_4",
    .udesc   = "Cycles at least 4 micro-op is executed from any thread on physical core.",
    .ucode   = 0x0200ull | (0x4 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_GE_1",
    .udesc   = "Cycles where at least 1 uop was executed per-thread",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_GE_2",
    .udesc   = "Cycles where at least 2 uops were executed per-thread",
    .ucode   = 0x0100ull | (0x2 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_GE_3",
    .udesc   = "Cycles where at least 3 uops were executed per-thread",
    .ucode   = 0x0100ull | (0x3 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "CYCLES_GE_4",
    .udesc   = "Cycles where at least 4 uops were executed per-thread",
    .ucode   = 0x0100ull | (0x4 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "STALLS",
    .udesc   = "Counts number of cycles no uops were dispatched to be executed on this thread.",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_INV_BIT) | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_I | _INTEL_X86_ATTR_C,
  },
  { .uname   = "STALL_CYCLES",
    .udesc   = "This event is deprecated. Refer to new event UOPS_EXECUTED.STALLS",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_INV_BIT) | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_I | _INTEL_X86_ATTR_C,
  },
  { .uname   = "THREAD",
    .udesc   = "Counts the number of uops to be executed per-thread each cycle.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "X87",
    .udesc   = "Counts the number of x87 uops dispatched.",
    .ucode   = 0x1000ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
};

static const intel_x86_umask_t adl_glc_uops_issued[]={
  { .uname   = "ANY",
    .udesc   = "Uops that RAT issues to RS",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_DFL,
  },
};

static const intel_x86_umask_t adl_glc_uops_retired[]={
  { .uname   = "CYCLES",
    .udesc   = "Cycles with retired uop(s).",
    .ucode   = 0x0200ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_C,
  },
  { .uname   = "HEAVY",
    .udesc   = "Retired uops except the last uop of each instruction.",
    .ucode   = 0x0100ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "MS",
    .udesc   = "TBD",
    .ucode   = 0x0800ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "SLOTS",
    .udesc   = "Retirement slots used.",
    .ucode   = 0x0200ull,
    .uflags  = INTEL_X86_NCOMBO,
  },
  { .uname   = "STALLS",
    .udesc   = "Cycles without actually retired uops.",
    .ucode   = 0x0200ull | (0x1 << INTEL_X86_INV_BIT) | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_I | _INTEL_X86_ATTR_C,
  },
  { .uname   = "STALL_CYCLES",
    .udesc   = "This event is deprecated. Refer to new event UOPS_RETIRED.STALLS",
    .ucode   = 0x0200ull | (0x1 << INTEL_X86_INV_BIT) | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_NCOMBO,
    .modhw   = _INTEL_X86_ATTR_I | _INTEL_X86_ATTR_C,
  },
};

static const intel_x86_umask_t adl_glc_xq[]={
  { .uname   = "FULL_CYCLES",
    .udesc   = "Cycles the uncore cannot take further requests",
    .ucode   = 0x0100ull | (0x1 << INTEL_X86_CMASK_BIT),
    .uflags  = INTEL_X86_DFL,
    .modhw   = _INTEL_X86_ATTR_C,
  },
};

static const intel_x86_entry_t intel_adl_glc_pe[]={
  { .name   = "ARITH",
    .desc   = "This event is deprecated. Refer to new event ARITH.FPDIV_ACTIVE",
    .code   = 0x00b0,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_arith),
    .umasks = adl_glc_arith,
  },
  { .name   = "ASSISTS",
    .desc   = "Counts all microcode FP assists.",
    .code   = 0x00c1,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_assists),
    .umasks = adl_glc_assists,
  },
  { .name   = "BACLEARS",
    .desc   = "Clears due to Unknown Branches.",
    .code   = 0x0060,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_baclears),
    .umasks = adl_glc_baclears,
  },
  { .name   = "BR_INST_RETIRED",
    .desc   = "All branch instructions retired.",
    .code   = 0x00c4,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_br_inst_retired),
    .umasks = adl_glc_br_inst_retired,
  },
  { .name   = "BR_MISP_RETIRED",
    .desc   = "All mispredicted branch instructions retired.",
    .code   = 0x00c5,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_br_misp_retired),
    .umasks = adl_glc_br_misp_retired,
  },
  { .name   = "CORE_POWER",
    .desc   = "TBD",
    .code   = 0x0028,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_core_power),
    .umasks = adl_glc_core_power,
  },
  { .name   = "CPU_CLK_UNHALTED",
    .desc   = "Core cycles when the thread is not in halt state",
    .code   = 0x003c,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0x1ull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_cpu_clk_unhalted),
    .umasks = adl_glc_cpu_clk_unhalted,
  },
  { .name   = "CYCLE_ACTIVITY",
    .desc   = "Cycles while L2 cache miss demand load is outstanding.",
    .code   = 0x00a3,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_cycle_activity),
    .umasks = adl_glc_cycle_activity,
  },
  { .name   = "DECODE",
    .desc   = "Stalls caused by changing prefix length of the instruction.",
    .code   = 0x0087,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_decode),
    .umasks = adl_glc_decode,
  },
  { .name   = "DSB2MITE_SWITCHES",
    .desc   = "DSB-to-MITE switch true penalty cycles.",
    .code   = 0x0061,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_dsb2mite_switches),
    .umasks = adl_glc_dsb2mite_switches,
  },
  { .name   = "DTLB_LOAD_MISSES",
    .desc   = "Page walks completed due to a demand data load to a 4K page.",
    .code   = 0x0012,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_dtlb_load_misses),
    .umasks = adl_glc_dtlb_load_misses,
  },
  { .name   = "DTLB_STORE_MISSES",
    .desc   = "Page walks completed due to a demand data store to a 4K page.",
    .code   = 0x0013,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_dtlb_store_misses),
    .umasks = adl_glc_dtlb_store_misses,
  },
  { .name   = "EXE_ACTIVITY",
    .desc   = "Cycles total of 1 uop is executed on all ports and Reservation Station was not empty.",
    .code   = 0x00a6,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_exe_activity),
    .umasks = adl_glc_exe_activity,
  },
  { .name   = "FP_ARITH_DISPATCHED",
    .desc   = "TBD",
    .code   = 0x00b3,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_fp_arith_dispatched),
    .umasks = adl_glc_fp_arith_dispatched,
  },
  { .name   = "FP_ARITH_INST_RETIRED",
    .desc   = "Counts number of SSE/AVX computational scalar double precision floating-point instructions retired; some instructions will count twice as noted below.  Each count represents 1 computational operation. Applies to SSE* and AVX* scalar double precision floating-point instructions",
    .code   = 0x00c7,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_fp_arith_inst_retired),
    .umasks = adl_glc_fp_arith_inst_retired,
  },
  { .name   = "FRONTEND_RETIRED",
    .desc   = "Retired Instructions who experienced a critical DSB miss.",
    .code   = 0x01c6,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_FRONTEND | INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_frontend_retired),
    .umasks = adl_glc_frontend_retired,
  },
  { .name   = "ICACHE_DATA",
    .desc   = "Cycles where a code fetch is stalled due to L1 instruction cache miss.",
    .code   = 0x0080,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_icache_data),
    .umasks = adl_glc_icache_data,
  },
  { .name   = "ICACHE_TAG",
    .desc   = "Cycles where a code fetch is stalled due to L1 instruction cache tag miss.",
    .code   = 0x0083,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_icache_tag),
    .umasks = adl_glc_icache_tag,
  },
  { .name   = "IDQ",
    .desc   = "Cycles MITE is delivering any Uop",
    .code   = 0x0079,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_idq),
    .umasks = adl_glc_idq,
  },
  { .name   = "IDQ_BUBBLES",
    .desc   = "Uops not delivered by IDQ when backend of the machine is not stalled",
    .equiv  = "IDQ_UOPS_NOT_DELIVERED",
    .code   = 0x009c,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_idq_uops_not_delivered),
    .umasks = adl_glc_idq_uops_not_delivered,
  },
  { .name   = "IDQ_UOPS_NOT_DELIVERED",
    .desc   = "Uops not delivered by IDQ when backend of the machine is not stalled [This event is alias to IDQ_BUBBLES.CORE]",
    .code   = 0x009c,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_idq_uops_not_delivered),
    .umasks = adl_glc_idq_uops_not_delivered,
  },
  { .name   = "INST_DECODED",
    .desc   = "Instruction decoders utilized in a cycle",
    .code   = 0x0075,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_inst_decoded),
    .umasks = adl_glc_inst_decoded,
  },
  { .name   = "INST_RETIRED",
    .desc   = "Number of instructions retired",
    .code   = 0x00c0,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0x1ull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_inst_retired),
    .umasks = adl_glc_inst_retired,
  },
  { .name   = "INSTRUCTION_RETIRED",
    .desc   = "Number of instructions at retirement",
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0x1000000ffull,
    .code = 0xc0,
  },
  { .name   = "INSTRUCTIONS_RETIRED",
    .desc   = "Number of instructions at retirement",
    .modmsk = INTEL_V5_ATTRS,
    .equiv = "INSTRUCTION_RETIRED",
    .cntmsk = 0x1000000ffull,
    .code = 0xc0,
  },
  { .name   = "INT_MISC",
    .desc   = "Core cycles the allocator was stalled due to recovery from earlier clear event for this thread",
    .code   = 0x00ad,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_int_misc),
    .umasks = adl_glc_int_misc,
  },
  { .name   = "INT_VEC_RETIRED",
    .desc   = "integer ADD, SUB, SAD 128-bit vector instructions.",
    .code   = 0x00e7,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_int_vec_retired),
    .umasks = adl_glc_int_vec_retired,
  },
  { .name   = "ITLB_MISSES",
    .desc   = "Code miss in all TLB levels causes a page walk that completes. (4K)",
    .code   = 0x0011,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_itlb_misses),
    .umasks = adl_glc_itlb_misses,
  },
  { .name   = "L1D",
    .desc   = "Counts the number of cache lines replaced in L1 data cache.",
    .code   = 0x0051,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_l1d),
    .umasks = adl_glc_l1d,
  },
  { .name   = "L1D_PEND_MISS",
    .desc   = "Number of L1D misses that are outstanding",
    .code   = 0x0048,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_l1d_pend_miss),
    .umasks = adl_glc_l1d_pend_miss,
  },
  { .name   = "L2_LINES_IN",
    .desc   = "L2 cache lines filling L2",
    .code   = 0x0025,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_l2_lines_in),
    .umasks = adl_glc_l2_lines_in,
  },
  { .name   = "L2_LINES_OUT",
    .desc   = "Cache lines that have been L2 hardware prefetched but not used by demand accesses",
    .code   = 0x0026,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_l2_lines_out),
    .umasks = adl_glc_l2_lines_out,
  },
  { .name   = "L2_REQUEST",
    .desc   = "Demand Data Read miss L2 cache",
    .equiv  = "L2_RQSTS",
    .code   = 0x0024,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_l2_rqsts),
    .umasks = adl_glc_l2_rqsts,
  },
  { .name   = "L2_RQSTS",
    .desc   = "Demand Data Read miss L2 cache",
    .code   = 0x0024,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_l2_rqsts),
    .umasks = adl_glc_l2_rqsts,
  },
  { .name   = "LD_BLOCKS",
    .desc   = "False dependencies in MOB due to partial compare on address.",
    .code   = 0x0003,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_ld_blocks),
    .umasks = adl_glc_ld_blocks,
  },
  { .name   = "LOAD_HIT_PREFETCH",
    .desc   = "Counts the number of demand load dispatches that hit L1D fill buffer (FB) allocated for software prefetch.",
    .code   = 0x004c,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_load_hit_prefetch),
    .umasks = adl_glc_load_hit_prefetch,
  },
  { .name   = "LONGEST_LAT_CACHE",
    .desc   = "Core-originated cacheable requests that missed L3  (Except hardware prefetches to the L3)",
    .code   = 0x002e,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_longest_lat_cache),
    .umasks = adl_glc_longest_lat_cache,
  },
  { .name   = "LSD",
    .desc   = "Cycles Uops delivered by the LSD, but didn't come from the decoder.",
    .code   = 0x00a8,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_lsd),
    .umasks = adl_glc_lsd,
  },
  { .name   = "MACHINE_CLEARS",
    .desc   = "Number of machine clears (nukes) of any type.",
    .code   = 0x00c3,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_machine_clears),
    .umasks = adl_glc_machine_clears,
  },
  { .name   = "MEMORY_ACTIVITY",
    .desc   = "Cycles while L1 cache miss demand load is outstanding.",
    .code   = 0x0047,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_memory_activity),
    .umasks = adl_glc_memory_activity,
  },
  { .name   = "MEM_INST_RETIRED",
    .desc   = "Retired load instructions that miss the STLB.",
    .code   = 0x00d0,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_mem_inst_retired),
    .umasks = adl_glc_mem_inst_retired,
  },
  { .name   = "MEM_LOAD_COMPLETED",
    .desc   = "Completed demand load uops that miss the L1 d-cache.",
    .code   = 0x0043,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_mem_load_completed),
    .umasks = adl_glc_mem_load_completed,
  },
  { .name   = "MEM_LOAD_L3_HIT_RETIRED",
    .desc   = "Retired load instructions whose data sources were L3 hit and cross-core snoop missed in on-pkg core cache.",
    .code   = 0x00d2,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_mem_load_l3_hit_retired),
    .umasks = adl_glc_mem_load_l3_hit_retired,
  },
  { .name   = "MEM_LOAD_L3_MISS_RETIRED",
    .desc   = "Retired load instructions which data sources missed L3 but serviced from local dram",
    .code   = 0x00d3,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_mem_load_l3_miss_retired),
    .umasks = adl_glc_mem_load_l3_miss_retired,
  },
  { .name   = "MEM_LOAD_MISC_RETIRED",
    .desc   = "Retired instructions with at least 1 uncacheable load or lock.",
    .code   = 0x00d4,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_mem_load_misc_retired),
    .umasks = adl_glc_mem_load_misc_retired,
  },
  { .name   = "MEM_LOAD_RETIRED",
    .desc   = "Retired load instructions with L1 cache hits as data sources",
    .code   = 0x00d1,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_mem_load_retired),
    .umasks = adl_glc_mem_load_retired,
  },
  { .name   = "MEM_STORE_RETIRED",
    .desc   = "TBD",
    .code   = 0x0044,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_mem_store_retired),
    .umasks = adl_glc_mem_store_retired,
  },
  { .name   = "MEM_TRANS_RETIRED",
    .desc   = "Counts randomly selected loads when the latency from first dispatch to completion is greater latency threshold in ldlat=",
    .code   = 0x01cd,
    .modmsk = INTEL_V4_ATTRS | _INTEL_X86_ATTR_LDLAT,
    .cntmsk = 0xfeull,
    .ngrp   = 1,
    .flags  = INTEL_X86_PEBS,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_mem_trans_retired),
    .umasks = adl_glc_mem_trans_retired,
  },
  { .name   = "MEM_UOP_RETIRED",
    .desc   = "Retired memory uops for any access",
    .code   = 0x00e5,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_mem_uop_retired),
    .umasks = adl_glc_mem_uop_retired,
  },
  { .name   = "MISC2_RETIRED",
    .desc   = "LFENCE instructions retired",
    .code   = 0x00e0,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_misc2_retired),
    .umasks = adl_glc_misc2_retired,
  },
  { .name   = "MISC_RETIRED",
    .desc   = "Increments whenever there is an update to the LBR array.",
    .code   = 0x00cc,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_misc_retired),
    .umasks = adl_glc_misc_retired,
  },
  { .name   = "OCR0",
    .desc   = "Counts demand data reads that were not supplied by the L3 cache.",
    .code   = 0x012a,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_NHM_OFFCORE,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_ocr),
    .umasks = adl_glc_ocr,
  },
  { .name   = "OCR1",
    .desc   = "Counts demand data reads that were not supplied by the L3 cache.",
    .code   = 0x012b,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_NHM_OFFCORE,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_ocr),
    .umasks = adl_glc_ocr,
  },
  { .name   = "OFFCORE_REQUESTS",
    .desc   = "Demand Data Read requests sent to uncore",
    .code   = 0x0021,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_offcore_requests),
    .umasks = adl_glc_offcore_requests,
  },
  { .name   = "OFFCORE_REQUESTS_OUTSTANDING",
    .desc   = "Cycles where at least 1 outstanding demand data read request is pending.",
    .code   = 0x0020,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_offcore_requests_outstanding),
    .umasks = adl_glc_offcore_requests_outstanding,
  },
  { .name   = "RESOURCE_STALLS",
    .desc   = "Counts cycles where the pipeline is stalled due to serializing operations.",
    .code   = 0x00a2,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_resource_stalls),
    .umasks = adl_glc_resource_stalls,
  },
  { .name   = "RS",
    .desc   = "Cycles when Reservation Station (RS) is empty for the thread.",
    .code   = 0x00a5,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_rs),
    .umasks = adl_glc_rs,
  },
  { .name   = "SQ_MISC",
    .desc   = "Counts bus locks, accounts for cache line split locks and UC locks.",
    .code   = 0x002c,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_sq_misc),
    .umasks = adl_glc_sq_misc,
  },
  { .name   = "SW_PREFETCH_ACCESS",
    .desc   = "Number of PREFETCHNTA instructions executed.",
    .code   = 0x0040,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_sw_prefetch_access),
    .umasks = adl_glc_sw_prefetch_access,
  },
  { .name   = "TOPDOWN",
    .desc   = "Topdown events using PERF_METRICS support",
    .code   = 0x0000,
    .modmsk = INTEL_FIXED2_ATTRS,
    .cntmsk = 0x800000000ull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_topdown),
    .umasks = adl_glc_topdown,
  },
  { .name   = "UNHALTED_CORE_CYCLES",
    .desc   = "Count core clock cycles whenever the clock signal on the specific core is running (not halted)",
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0x20000000full,
    .code = 0x3c,
  },
  { .name   = "UNHALTED_REFERENCE_CYCLES",
    .desc   = "Unhalted reference cycles",
    .modmsk = INTEL_FIXED3_ATTRS,
    .cntmsk = 0x400000000ull,
    .code = 0x0300, /* pseudo encoding */
    .flags = INTEL_X86_FIXED,
  },
  { .name   = "UOPS_DECODED",
    .desc   = "TBD",
    .code   = 0x0076,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_uops_decoded),
    .umasks = adl_glc_uops_decoded,
  },
  { .name   = "UOPS_DISPATCHED",
    .desc   = "Uops executed on port 0",
    .code   = 0x00b2,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_uops_dispatched),
    .umasks = adl_glc_uops_dispatched,
  },
  { .name   = "UOPS_EXECUTED",
    .desc   = "Cycles where at least 1 uop was executed per-thread",
    .code   = 0x00b1,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_uops_executed),
    .umasks = adl_glc_uops_executed,
  },
  { .name   = "UOPS_ISSUED",
    .desc   = "Uops that RAT issues to RS",
    .code   = 0x00ae,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_uops_issued),
    .umasks = adl_glc_uops_issued,
  },
  { .name   = "UOPS_RETIRED",
    .desc   = "Retired uops except the last uop of each instruction.",
    .code   = 0x00c2,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xffull,
    .ngrp   = 1,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_uops_retired),
    .umasks = adl_glc_uops_retired,
  },
  { .name   = "XQ",
    .desc   = "Cycles the uncore cannot take further requests",
    .code   = 0x002d,
    .modmsk = INTEL_V5_ATTRS,
    .cntmsk = 0xfull,
    .ngrp   = 1,
    .flags  = INTEL_X86_SPEC,
    .numasks= LIBPFM_ARRAY_SIZE(adl_glc_xq),
    .umasks = adl_glc_xq,
  },
};
/* 64 events available */
